"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGlobalExecutionKpiWithAuth = getGlobalExecutionKpiWithAuth;
exports.getRuleExecutionKPI = getRuleExecutionKPI;
var _authorization = require("../../authorization");
var _audit_events = require("../common/audit_events");
var _get_execution_log_aggregation = require("../../lib/get_execution_log_aggregation");
var _common = require("../common");
var _get_rule = require("../../application/rule/methods/get/get_rule");
var _saved_objects = require("../../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getRuleExecutionKPI(context, {
  id,
  dateStart,
  dateEnd,
  filter
}) {
  var _context$auditLogger2;
  context.logger.debug(`getRuleExecutionKPI(): getting execution KPI for rule ${id}`);
  const rule = await (0, _get_rule.getRule)(context, {
    id,
    includeLegacyId: true
  });
  try {
    // Make sure user has access to this rule
    await context.authorization.ensureAuthorized({
      ruleTypeId: rule.alertTypeId,
      consumer: rule.consumer,
      operation: _authorization.ReadOperations.GetRuleExecutionKPI,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.GET_RULE_EXECUTION_KPI,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id,
        name: rule.name
      },
      error
    }));
    throw error;
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.GET_RULE_EXECUTION_KPI,
    savedObject: {
      type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
      id,
      name: rule.name
    }
  }));

  // default duration of instance summary is 60 * rule interval
  const dateNow = new Date();
  const parsedDateStart = (0, _common.parseDate)(dateStart, 'dateStart', dateNow);
  const parsedDateEnd = (0, _common.parseDate)(dateEnd, 'dateEnd', dateNow);
  const eventLogClient = await context.getEventLogClient();
  try {
    const aggResult = await eventLogClient.aggregateEventsBySavedObjectIds(_saved_objects.RULE_SAVED_OBJECT_TYPE, [id], {
      start: parsedDateStart.toISOString(),
      end: parsedDateEnd.toISOString(),
      aggs: (0, _get_execution_log_aggregation.getExecutionKPIAggregation)(filter)
    }, rule.legacyId !== null ? [rule.legacyId] : undefined);
    return (0, _get_execution_log_aggregation.formatExecutionKPIResult)(aggResult);
  } catch (err) {
    context.logger.debug(`rulesClient.getRuleExecutionKPI(): error searching execution KPI for rule ${id}: ${err.message}`);
    throw err;
  }
}
async function getGlobalExecutionKpiWithAuth(context, {
  dateStart,
  dateEnd,
  filter,
  namespaces
}) {
  var _context$auditLogger4;
  context.logger.debug(`getGlobalExecutionLogWithAuth(): getting global execution log`);
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter({
      authorizationEntity: _authorization.AlertingAuthorizationEntity.Alert,
      filterOpts: {
        type: _authorization.AlertingAuthorizationFilterType.KQL,
        fieldNames: {
          ruleTypeId: 'kibana.alert.rule.rule_type_id',
          consumer: 'kibana.alert.rule.consumer'
        }
      }
    });
  } catch (error) {
    var _context$auditLogger3;
    (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.GET_GLOBAL_EXECUTION_KPI,
      error
    }));
    throw error;
  }
  (_context$auditLogger4 = context.auditLogger) === null || _context$auditLogger4 === void 0 ? void 0 : _context$auditLogger4.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.GET_GLOBAL_EXECUTION_KPI
  }));
  const dateNow = new Date();
  const parsedDateStart = (0, _common.parseDate)(dateStart, 'dateStart', dateNow);
  const parsedDateEnd = (0, _common.parseDate)(dateEnd, 'dateEnd', dateNow);
  const eventLogClient = await context.getEventLogClient();
  try {
    const aggResult = await eventLogClient.aggregateEventsWithAuthFilter(_saved_objects.RULE_SAVED_OBJECT_TYPE, authorizationTuple.filter, {
      start: parsedDateStart.toISOString(),
      end: parsedDateEnd.toISOString(),
      aggs: (0, _get_execution_log_aggregation.getExecutionKPIAggregation)(filter)
    }, namespaces);
    return (0, _get_execution_log_aggregation.formatExecutionKPIResult)(aggResult);
  } catch (err) {
    context.logger.debug(`rulesClient.getGlobalExecutionKpiWithAuth(): error searching global execution KPI: ${err.message}`);
    throw err;
  }
}