"use strict";
/**
 * MIT License
 *
 * Copyright (c) 2020-present, Elastic NV
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.runLocal = void 0;
const child_process_1 = require("child_process");
const promises_1 = require("fs/promises");
const fs_1 = require("fs");
const os_1 = require("os");
const unzip_stream_1 = require("unzip-stream");
const colors_1 = require("kleur/colors");
const path_1 = require("path");
const url_1 = require("url");
async function unzipFile(zipPath, destination) {
    return new Promise((resolve, reject) => {
        (0, fs_1.createReadStream)(zipPath)
            .pipe((0, unzip_stream_1.Extract)({ path: destination }))
            .on('close', resolve)
            .on('error', err => reject(new Error(`failed to extract zip ${zipPath} : ${err.message}`)));
    });
}
async function runNpmInstall(directory) {
    return new Promise((resolve, reject) => {
        const flags = [
            '--no-audit',
            '--no-update-notifier',
            '--no-fund',
            '--package-lock=false',
            '--progress=false', // no need to display progress
        ];
        const npmInstall = (0, child_process_1.spawn)('npm', ['install', ...flags], {
            cwd: directory,
            stdio: 'ignore',
        });
        npmInstall.on('close', code => {
            if (code === 0) {
                resolve();
            }
            else {
                reject(new Error(`npm install failed with exit code ${code}`));
            }
        });
        npmInstall.on('error', err => reject(new Error(`failed to setup: ${err.message}`)));
    });
}
async function runTest(directory, schema) {
    return new Promise((resolve, reject) => {
        const runTest = (0, child_process_1.spawn)('npx', [
            '@elastic/synthetics',
            '.',
            '--playwright-options',
            JSON.stringify(schema.playwrightOptions),
            '--params',
            JSON.stringify(schema.params),
        ], {
            cwd: directory,
            stdio: 'inherit',
        });
        runTest.on('close', resolve);
        runTest.on('error', err => {
            reject(new Error(`Failed to execute @elastic/synthetics : ${err.message}`));
        });
    });
}
async function writePkgJSON(dir, synthPath) {
    const packageJsonContent = {
        name: 'project-journey',
        private: 'true',
        dependencies: {
            '@elastic/synthetics': (0, url_1.pathToFileURL)(synthPath),
        },
    };
    await (0, promises_1.writeFile)((0, path_1.join)(dir, 'package.json'), JSON.stringify(packageJsonContent, null, 2), 'utf-8');
}
async function extract(schema, zipPath, unzipPath) {
    if (schema.type !== 'browser') {
        return;
    }
    const content = schema.content;
    await (0, promises_1.writeFile)(zipPath, content, 'base64');
    await unzipFile(zipPath, unzipPath);
}
async function runLocal(schemas) {
    // lookup installed bin path of a node module
    const resolvedPath = (0, child_process_1.execFileSync)('which', ['elastic-synthetics'], {
        encoding: 'utf8',
    }).trim();
    const synthPath = resolvedPath.replace((0, path_1.join)('bin', 'elastic-synthetics'), (0, path_1.join)('lib', 'node_modules', '@elastic/synthetics'));
    const rand = Date.now();
    const zipPath = (0, path_1.join)((0, os_1.tmpdir)(), `synthetics-zip-${rand}.zip`);
    const unzipPath = (0, path_1.join)((0, os_1.tmpdir)(), `synthetics-unzip-${rand}`);
    try {
        for (const schema of schemas) {
            await extract(schema, zipPath, unzipPath);
        }
        await writePkgJSON(unzipPath, synthPath);
        await runNpmInstall(unzipPath);
        // TODO: figure out a way to collect all params and Playwright options
        await runTest(unzipPath, schemas[0]);
    }
    catch (e) {
        throw (0, colors_1.red)(`Aborted: ${e.message}`);
    }
    finally {
        await (0, promises_1.rm)(zipPath, { recursive: true, force: true });
        await (0, promises_1.rm)(unzipPath, { recursive: true, force: true });
    }
}
exports.runLocal = runLocal;
//# sourceMappingURL=run-local.js.map