"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createConversationRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _event_based_telemetry = require("../../lib/telemetry/event_based_telemetry");
var _utils = require("../utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createConversationRoute = router => {
  router.versioned.post({
    access: 'public',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_CONVERSATIONS_URL,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ConversationCreateProps)
      }
    }
  }, async (context, request, response) => {
    const assistantResponse = (0, _utils.buildResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
      // Perform license and authenticated user checks
      const checkResponse = await (0, _helpers.performChecks)({
        context: ctx,
        request,
        response
      });
      if (!checkResponse.isSuccess) {
        return checkResponse.response;
      }
      const dataClient = await ctx.elasticAssistant.getAIAssistantConversationsDataClient();
      const createdConversation = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createConversation({
        conversation: request.body
      }));
      if (createdConversation == null) {
        return assistantResponse.error({
          body: `conversation with title: "${request.body.title}" was not created`,
          statusCode: 400
        });
      }
      if (createdConversation.title.startsWith('[Duplicate]')) {
        var _createdConversation$, _firstUserMessage$use, _checkResponse$curren, _firstUserMessage$use2, _checkResponse$curren2;
        const telemetry = ctx.elasticAssistant.telemetry;
        const firstUserMessage = (_createdConversation$ = createdConversation.messages) === null || _createdConversation$ === void 0 ? void 0 : _createdConversation$.find(m => m.role === 'user');
        // if the user duplicates a conversation that they did not own
        // and duplicates it again, we will still find them to be the owner since
        // the messages will still be of the original owner.
        // I think that's fine? We don't have another way to tell since
        // no createdBy is sent to the create api
        const isSourceConversationOwner = (firstUserMessage === null || firstUserMessage === void 0 ? void 0 : (_firstUserMessage$use = firstUserMessage.user) === null || _firstUserMessage$use === void 0 ? void 0 : _firstUserMessage$use.name) === ((_checkResponse$curren = checkResponse.currentUser) === null || _checkResponse$curren === void 0 ? void 0 : _checkResponse$curren.username) || (firstUserMessage === null || firstUserMessage === void 0 ? void 0 : (_firstUserMessage$use2 = firstUserMessage.user) === null || _firstUserMessage$use2 === void 0 ? void 0 : _firstUserMessage$use2.id) === ((_checkResponse$curren2 = checkResponse.currentUser) === null || _checkResponse$curren2 === void 0 ? void 0 : _checkResponse$curren2.profile_uid);
        telemetry.reportEvent(_event_based_telemetry.CONVERSATION_DUPLICATED_EVENT.eventType, {
          isSourceConversationOwner
        });
      }
      return response.ok({
        body: _elasticAssistantCommon.ConversationResponse.parse(createdConversation)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return assistantResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.createConversationRoute = createConversationRoute;