"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDelayedVisibility = void 0;
var _react = require("react");
var _onboarding_card_panel = require("../onboarding_card_panel.styles");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This hook manages a delayed visibility state (`isVisible`) to support smooth animations
 * during expand and collapse transitions. It delays the `isVisible` state change when
 * collapsing, allowing animations to complete before unmounting the component.
 *
 * @param {Object} params - Parameters object.
 * @param {boolean} params.isExpanded - Controls whether the component should be expanded
 * (fully visible) or collapsed (delayed unmount).
 *
 * @returns {boolean} isVisible - True if the component should be visible, false otherwise.
 */
const useDelayedVisibility = ({
  isExpanded
}) => {
  const [isVisible, setIsVisible] = (0, _react.useState)(isExpanded);
  (0, _react.useEffect)(() => {
    let timeoutId;

    // Delay visibility toggle off to allow exit animations to complete
    if (!isExpanded) {
      timeoutId = setTimeout(() => setIsVisible(false), _onboarding_card_panel.HEIGHT_ANIMATION_DURATION);
    } else {
      setIsVisible(true);
    }
    return () => clearTimeout(timeoutId);
  }, [isExpanded]);
  return isVisible;
};
exports.useDelayedVisibility = useDelayedVisibility;