"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TELEMETRY_TASK_TYPE = exports.TASK_ID = exports.SCHEDULE = void 0;
exports.initializeReportingTelemetryTask = initializeReportingTelemetryTask;
exports.scheduleReportingTelemetry = scheduleReportingTelemetry;
exports.telemetryTaskRunner = telemetryTaskRunner;
var _get_telemetry_from_kibana = require("./lib/get_telemetry_from_kibana");
var _task_state = require("./task_state");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TELEMETRY_TASK_TYPE = exports.TELEMETRY_TASK_TYPE = 'reporting_telemetry';
const TASK_ID = exports.TASK_ID = `Reporting-${TELEMETRY_TASK_TYPE}`;
const SCHEDULE = exports.SCHEDULE = {
  interval: '1d'
};
function initializeReportingTelemetryTask(logger, core, taskManager) {
  registerReportingTelemetryTask(logger, core, taskManager);
}
function registerReportingTelemetryTask(logger, core, taskManager) {
  taskManager.registerTaskDefinitions({
    [TELEMETRY_TASK_TYPE]: {
      title: 'Reporting snapshot telemetry fetch task',
      timeout: '5m',
      stateSchemaByVersion: _task_state.stateSchemaByVersion,
      createTaskRunner: telemetryTaskRunner(logger, core)
    }
  });
}
function scheduleReportingTelemetry(logger, taskManager) {
  if (taskManager) {
    taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TELEMETRY_TASK_TYPE,
      state: _task_state.emptyState,
      params: {},
      schedule: SCHEDULE
    }).catch(e => logger.error(`Error scheduling ${TASK_ID}, received ${e.message}`));
  }
}
function telemetryTaskRunner(logger, core) {
  return ({
    taskInstance
  }) => {
    const state = taskInstance.state;
    const getServices = () => core.getStartServices().then(([coreStart]) => {
      return {
        savedObjectIndex: coreStart.savedObjects.getIndexForType(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE),
        esClient: coreStart.elasticsearch.client.asInternalUser
      };
    });
    return {
      async run() {
        const {
          esClient,
          savedObjectIndex
        } = await getServices();
        return (0, _get_telemetry_from_kibana.getTotalCountAggregations)({
          esClient,
          index: savedObjectIndex,
          logger
        }).then(totalCountAggregations => {
          const hasErrors = totalCountAggregations.hasErrors;
          const errorMessages = [totalCountAggregations.errorMessage].filter(message => message !== undefined);
          const updatedState = {
            has_errors: hasErrors,
            ...(totalCountAggregations.errorMessage && {
              error_messages: [errorMessages]
            }),
            runs: (state.runs || 0) + 1,
            number_of_scheduled_reports: totalCountAggregations.number_of_scheduled_reports,
            number_of_enabled_scheduled_reports: totalCountAggregations.number_of_enabled_scheduled_reports,
            number_of_scheduled_reports_by_type: totalCountAggregations.number_of_scheduled_reports_by_type,
            number_of_enabled_scheduled_reports_by_type: totalCountAggregations.number_of_enabled_scheduled_reports_by_type,
            number_of_scheduled_reports_with_notifications: totalCountAggregations.number_of_scheduled_reports_with_notifications
          };
          return {
            state: updatedState,
            // Useful for setting a schedule for the old tasks that don't have one
            // or to update the schedule if ever the frequency changes in code
            schedule: SCHEDULE
          };
        }).catch(errMsg => {
          logger.warn(`Error executing reporting telemetry task: ${errMsg}`);
          return {
            state: _task_state.emptyState,
            // Useful for setting a schedule for the old tasks that don't have one
            // or to update the schedule if ever the frequency changes in code
            schedule: SCHEDULE
          };
        });
      }
    };
  };
}