"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopInfluencers = getTopInfluencers;
var _explorer = require("../../../common/constants/explorer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTopInfluencers(mlClient, {
  jobIds,
  earliestMs,
  latestMs,
  influencers = [],
  influencersFilterQuery,
  maxFieldValues,
  perPage,
  page
}) {
  var _influencerFieldNames, _resp$aggregations, _resp$aggregations$in;
  // Build the criteria to use in the bool filter part of the request.
  // Adds criteria for the time range plus any specified job IDs.
  const boolCriteria = [{
    range: {
      timestamp: {
        gte: earliestMs,
        lte: latestMs,
        format: 'epoch_millis'
      }
    }
  }, {
    range: {
      influencer_score: {
        gt: 0
      }
    }
  }];
  if (jobIds && jobIds.length > 0 && !(jobIds.length === 1 && jobIds[0] === '*')) {
    boolCriteria.push({
      terms: {
        job_id: jobIds
      }
    });
  }
  if (influencersFilterQuery !== undefined) {
    boolCriteria.push(influencersFilterQuery);
  }

  // Add a should query to filter for each of the specified influencers.
  if (influencers.length > 0) {
    boolCriteria.push({
      bool: {
        should: influencers.map(inf => ({
          bool: {
            must: [{
              term: {
                influencer_field_name: inf.fieldName
              }
            }, {
              term: {
                influencer_field_value: inf.fieldValue
              }
            }]
          }
        })),
        minimum_should_match: 1
      }
    });
  }
  const maxValues = typeof maxFieldValues === 'number' && maxFieldValues > 0 ? maxFieldValues : _explorer.ANOMALY_SWIM_LANE_HARD_LIMIT;
  const pageSize = typeof perPage === 'number' && perPage > 0 ? perPage : 10;
  const pageIndex = typeof page === 'number' && page > 0 ? page : 1; // 1-based
  const from = (pageIndex - 1) * pageSize;
  const resp = await mlClient.anomalySearch({
    size: 0,
    query: {
      bool: {
        filter: [{
          query_string: {
            query: 'result_type:influencer',
            analyze_wildcard: false
          }
        }, {
          bool: {
            must: boolCriteria
          }
        }]
      }
    },
    aggs: {
      influencerFieldNames: {
        terms: {
          field: 'influencer_field_name',
          size: 5,
          order: {
            maxAnomalyScore: 'desc'
          }
        },
        aggs: {
          maxAnomalyScore: {
            max: {
              field: 'influencer_score'
            }
          },
          influencerFieldValues: {
            terms: {
              field: 'influencer_field_value',
              size: maxValues,
              order: {
                maxAnomalyScore: 'desc'
              }
            },
            aggs: {
              bucketTruncate: {
                bucket_sort: {
                  from,
                  size: pageSize
                }
              },
              maxAnomalyScore: {
                max: {
                  field: 'influencer_score'
                }
              },
              sumAnomalyScore: {
                sum: {
                  field: 'influencer_score'
                }
              }
            }
          }
        }
      }
    }
  }, jobIds);
  const result = {};
  const fieldNameBuckets = (_influencerFieldNames = (_resp$aggregations = resp.aggregations) === null || _resp$aggregations === void 0 ? void 0 : (_resp$aggregations$in = _resp$aggregations.influencerFieldNames) === null || _resp$aggregations$in === void 0 ? void 0 : _resp$aggregations$in.buckets) !== null && _influencerFieldNames !== void 0 ? _influencerFieldNames : [];
  for (const nameBucket of fieldNameBuckets) {
    var _nameBucket$influence, _nameBucket$influence2;
    const fieldName = nameBucket.key;
    const valueBuckets = (_nameBucket$influence = (_nameBucket$influence2 = nameBucket.influencerFieldValues) === null || _nameBucket$influence2 === void 0 ? void 0 : _nameBucket$influence2.buckets) !== null && _nameBucket$influence !== void 0 ? _nameBucket$influence : [];
    result[fieldName] = valueBuckets.map(b => {
      var _b$maxAnomalyScore$va, _b$maxAnomalyScore, _b$sumAnomalyScore$va, _b$sumAnomalyScore;
      return {
        influencerFieldValue: b.key,
        maxAnomalyScore: (_b$maxAnomalyScore$va = (_b$maxAnomalyScore = b.maxAnomalyScore) === null || _b$maxAnomalyScore === void 0 ? void 0 : _b$maxAnomalyScore.value) !== null && _b$maxAnomalyScore$va !== void 0 ? _b$maxAnomalyScore$va : 0,
        sumAnomalyScore: (_b$sumAnomalyScore$va = (_b$sumAnomalyScore = b.sumAnomalyScore) === null || _b$sumAnomalyScore === void 0 ? void 0 : _b$sumAnomalyScore.value) !== null && _b$sumAnomalyScore$va !== void 0 ? _b$sumAnomalyScore$va : 0
      };
    });
  }
  return result;
}