"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformMetricsApiResponseToSnapshotResponse = void 0;
var _lodash = require("lodash");
var _constants = require("./constants");
var _apply_metadata_to_last_path = require("./apply_metadata_to_last_path");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getMetricValue = row => {
  if (!(0, _lodash.isNumber)(row.metric_0)) return null;
  const value = row.metric_0;
  return Number.isFinite(value) ? value : null;
};
const calculateMax = rows => {
  return (0, _lodash.max)(rows.map(getMetricValue)) || 0;
};
const calculateAvg = rows => {
  const values = rows.map(getMetricValue).filter(Number.isFinite);
  return (0, _lodash.sum)(values) / Math.max(values.length, 1);
};
const getLastValue = rows => {
  const row = (0, _lodash.last)(rows);
  if (!row) return null;
  return getMetricValue(row);
};
const transformMetricsApiResponseToSnapshotResponse = (options, snapshotRequest, metricsApiResponse) => {
  const nodes = metricsApiResponse.series.map(series => {
    var _series$keys, _series$metricsets, _last$label, _last;
    const node = {
      name: '',
      metrics: getMetrics(options, snapshotRequest, series),
      path: ((_series$keys = series.keys) !== null && _series$keys !== void 0 ? _series$keys : []).map(key => {
        return {
          value: key,
          label: key
        };
      })
    };
    const isNoData = node.metrics.every(m => m.value === null);
    const isAPMNode = (_series$metricsets = series.metricsets) === null || _series$metricsets === void 0 ? void 0 : _series$metricsets.includes('app');
    if (isNoData && isAPMNode) return null;
    const path = (0, _apply_metadata_to_last_path.applyMetadataToLastPath)(series, node, snapshotRequest);
    const name = (_last$label = (_last = (0, _lodash.last)(path)) === null || _last === void 0 ? void 0 : _last.label) !== null && _last$label !== void 0 ? _last$label : 'N/A';
    return {
      ...node,
      path,
      name
    };
  }).filter(n => n !== null);
  return {
    nodes,
    interval: metricsApiResponse.info.interval !== undefined ? `${metricsApiResponse.info.interval}s` : undefined
  };
};
exports.transformMetricsApiResponseToSnapshotResponse = transformMetricsApiResponseToSnapshotResponse;
const getMetrics = (options, snapshotRequest, series) => {
  return options.metrics.filter(m => m.id !== _constants.META_KEY).map(metric => {
    const name = metric.id;
    const metrics = series.rows.map(row => ({
      timestamp: row.timestamp,
      metric_0: (0, _lodash.get)(row, metric.id, null)
    }));
    const timeseries = snapshotRequest.includeTimeseries ? {
      id: name,
      columns: [{
        name: 'timestamp',
        type: 'date'
      }, {
        name: 'metric_0',
        type: 'number'
      }],
      rows: [...metrics]
    } : undefined;
    return {
      name,
      value: getLastValue(metrics),
      max: calculateMax(metrics),
      avg: calculateAvg(metrics),
      timeseries
    };
  });
};