"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkIntegrationFipsLooseCompatibility = checkIntegrationFipsLooseCompatibility;
exports.getNonFipsIntegrations = getNonFipsIntegrations;
exports.getRootIntegrations = getRootIntegrations;
exports.getRootPrivilegedDataStreams = getRootPrivilegedDataStreams;
exports.hasInstallServersInputs = hasInstallServersInputs;
exports.isRootPrivilegesRequired = isRootPrivilegesRequired;
var _lodash = require("lodash");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Return true if a package need Elastic Agent to be run as root/administrator
 */
function isRootPrivilegesRequired(packageInfo) {
  var _packageInfo$agent, _packageInfo$agent$pr, _packageInfo$data_str;
  return ((_packageInfo$agent = packageInfo.agent) === null || _packageInfo$agent === void 0 ? void 0 : (_packageInfo$agent$pr = _packageInfo$agent.privileges) === null || _packageInfo$agent$pr === void 0 ? void 0 : _packageInfo$agent$pr.root) || ((_packageInfo$data_str = packageInfo.data_streams) === null || _packageInfo$data_str === void 0 ? void 0 : _packageInfo$data_str.some(d => {
    var _d$agent, _d$agent$privileges;
    return (_d$agent = d.agent) === null || _d$agent === void 0 ? void 0 : (_d$agent$privileges = _d$agent.privileges) === null || _d$agent$privileges === void 0 ? void 0 : _d$agent$privileges.root;
  }));
}
function getRootPrivilegedDataStreams(packageInfo) {
  var _packageInfo$agent2, _packageInfo$agent2$p, _packageInfo$data_str2, _packageInfo$data_str3;
  if ((_packageInfo$agent2 = packageInfo.agent) !== null && _packageInfo$agent2 !== void 0 && (_packageInfo$agent2$p = _packageInfo$agent2.privileges) !== null && _packageInfo$agent2$p !== void 0 && _packageInfo$agent2$p.root) {
    return [];
  }
  return (_packageInfo$data_str2 = (_packageInfo$data_str3 = packageInfo.data_streams) === null || _packageInfo$data_str3 === void 0 ? void 0 : _packageInfo$data_str3.filter(d => {
    var _d$agent2, _d$agent2$privileges;
    return (_d$agent2 = d.agent) === null || _d$agent2 === void 0 ? void 0 : (_d$agent2$privileges = _d$agent2.privileges) === null || _d$agent2$privileges === void 0 ? void 0 : _d$agent2$privileges.root;
  }).map(ds => ({
    name: ds.name,
    title: ds.title
  }))) !== null && _packageInfo$data_str2 !== void 0 ? _packageInfo$data_str2 : [];
}
function getRootIntegrations(packagePolicies) {
  return (0, _lodash.uniqBy)(packagePolicies.map(policy => policy.package).filter(pkg => pkg && pkg.requires_root || false), pkg => pkg.name).map(pkg => ({
    name: pkg.name,
    title: pkg.title
  }));
}
const INSTALL_SERVERS_INPUTS = ['cloudbeat', 'apm', 'fleet-server'];
function hasInstallServersInputs(packagePolicies) {
  return packagePolicies.some(policy => policy.inputs.some(input => INSTALL_SERVERS_INPUTS.includes(input.type) || input.type.startsWith('cloudbeat')));
}

/**
 * Return true if a package is fips compatible.
 * Policy templates that have fips_compatible not defined are considered compatible.
 * Only `fips_compatible: false` is considered not compatible, except for OTel inputs
 * that are considered incompatible by default.
 */
function checkIntegrationFipsLooseCompatibility(integrationName, packageInfo) {
  var _packageInfo$policy_t;
  if (!(packageInfo !== null && packageInfo !== void 0 && packageInfo.policy_templates) || ((_packageInfo$policy_t = packageInfo.policy_templates) === null || _packageInfo$policy_t === void 0 ? void 0 : _packageInfo$policy_t.length) === 0) {
    return true;
  }
  if (packageInfo.policy_templates.find(p => p.name === integrationName && checkPolicyTemplateFipsCompatibility(p))) {
    return true;
  }
  return false;
}

/** Return true if a policy template is fips compatible.
 * It is fips compatible if it says so, or if it doesn't contain an OTel input.
 */
function checkPolicyTemplateFipsCompatibility(template) {
  if (template.fips_compatible !== undefined) {
    return template.fips_compatible;
  }
  if ('input' in template) {
    // Policy template in input package.
    return template.input !== _constants.OTEL_COLLECTOR_INPUT_TYPE;
  } else if ('inputs' in template) {
    var _template$inputs;
    // Policy templates in integration package.
    if (!((_template$inputs = template.inputs) !== null && _template$inputs !== void 0 && _template$inputs.find(input => input.type === _constants.OTEL_COLLECTOR_INPUT_TYPE))) {
      return true;
    }
  } else {
    // No inputs or input type.
    return true;
  }
  return false;
}

/**
 * Given a package policy list, get the list of integrations that are explicitly marked as not compatible with FIPS
 *
 */
function getNonFipsIntegrations(packagePolicies) {
  return (0, _lodash.uniqBy)(packagePolicies.map(policy => policy.package).filter(pkg => pkg && pkg.fips_compatible === false), pkg => pkg.name).map(pkg => ({
    name: pkg.name,
    title: pkg.title
  }));
}