/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch.search.aggregations.metrics;

import org.apache.lucene.search.Collector;
import org.apache.lucene.search.FieldDoc;
import org.apache.lucene.search.LeafCollector;
import org.apache.lucene.search.MultiCollector;
import org.apache.lucene.search.Scorable;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.ScoreMode;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopDocsCollector;
import org.apache.lucene.search.TopFieldCollector;
import org.apache.lucene.search.TopFieldCollectorManager;
import org.apache.lucene.search.TopFieldDocs;
import org.apache.lucene.search.TopScoreDocCollector;
import org.apache.lucene.search.TopScoreDocCollectorManager;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.search.MaxScoreCollector;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.search.TopDocsAndMaxScore;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.common.util.LongObjectPagedHashMap;
import org.elasticsearch.common.util.LongObjectPagedHashMap.Cursor;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.index.query.SearchExecutionContext;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationExecutionContext;
import org.elasticsearch.search.aggregations.Aggregator;
import org.elasticsearch.search.aggregations.InternalAggregation;
import org.elasticsearch.search.aggregations.LeafBucketCollector;
import org.elasticsearch.search.aggregations.LeafBucketCollectorBase;
import org.elasticsearch.search.aggregations.support.AggregationContext;
import org.elasticsearch.search.fetch.FetchSearchResult;
import org.elasticsearch.search.fetch.subphase.InnerHitsContext;
import org.elasticsearch.search.fetch.subphase.InnerHitsContext.InnerHitSubContext;
import org.elasticsearch.search.internal.SubSearchContext;
import org.elasticsearch.search.profile.ProfileResult;
import org.elasticsearch.search.rescore.RescoreContext;
import org.elasticsearch.search.sort.SortAndFormats;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;

class TopHitsAggregator extends MetricsAggregator {

    private static class Collectors {
        public final TopDocsCollector<?> topDocsCollector;
        public final MaxScoreCollector maxScoreCollector;
        public final Collector collector;

        Collectors(TopDocsCollector<?> topDocsCollector, MaxScoreCollector maxScoreCollector) {
            this.topDocsCollector = topDocsCollector;
            this.maxScoreCollector = maxScoreCollector;
            collector = MultiCollector.wrap(topDocsCollector, maxScoreCollector);
        }
    }

    private final BigArrays bigArrays;
    private final SubSearchContext subSearchContext;
    private final LongObjectPagedHashMap<Collectors> topDocsCollectors;
    private final List<ProfileResult> fetchProfiles;
    // this must be mutable so it can be closed/replaced on each call to getLeafCollector
    private LongObjectPagedHashMap<LeafCollector> leafCollectors;

    TopHitsAggregator(
        SubSearchContext subSearchContext,
        String name,
        AggregationContext context,
        Aggregator parent,
        Map<String, Object> metadata
    ) throws IOException {
        super(name, context, parent, metadata);
        this.bigArrays = context.bigArrays();
        this.subSearchContext = subSearchContext;
        this.topDocsCollectors = new LongObjectPagedHashMap<>(1, bigArrays);
        this.fetchProfiles = context.profiling() ? new ArrayList<>() : null;
    }

    @Override
    public ScoreMode scoreMode() {
        SortAndFormats sort = subSearchContext.sort();
        if (sort != null) {
            return sort.sort.needsScores() || subSearchContext.trackScores() ? ScoreMode.COMPLETE : ScoreMode.COMPLETE_NO_SCORES;
        } else {
            // sort by score
            return ScoreMode.COMPLETE;
        }
    }

    @Override
    public LeafBucketCollector getLeafCollector(AggregationExecutionContext aggCtx, LeafBucketCollector sub) throws IOException {
        // Create leaf collectors here instead of at the aggregator level. Otherwise in case this collector get invoked
        // when post collecting then we have already replaced the leaf readers on the aggregator level have already been
        // replaced with the next leaf readers and then post collection pushes docids of the previous segment, which
        // then causes assertions to trip or incorrect top docs to be computed.
        if (leafCollectors != null) {
            leafCollectors.close();
            leafCollectors = null; // set to null, just in case the new allocation below fails
        }
        leafCollectors = new LongObjectPagedHashMap<>(1, bigArrays);
        return new LeafBucketCollectorBase(sub, null) {

            // use the same Scorable for the leaf collectors
            ResettableScorable scorer = null;

            @Override
            public void setScorer(Scorable scorer) throws IOException {
                if (this.scorer != null) {
                    this.scorer.reset(scorer);
                    super.setScorer(scorer);
                } else {
                    this.scorer = new ResettableScorable(scorer);
                    super.setScorer(scorer);
                    for (Cursor<LeafCollector> leafCollector : leafCollectors) {
                        leafCollector.value.setScorer(scorer);
                    }
                }
            }

            @Override
            public void collect(int docId, long bucket) throws IOException {
                Collectors collectors = topDocsCollectors.get(bucket);
                if (collectors == null) {
                    SortAndFormats sort = subSearchContext.sort();
                    int topN = subSearchContext.from() + subSearchContext.size();
                    if (sort == null) {
                        for (RescoreContext rescoreContext : subSearchContext.rescore()) {
                            topN = Math.max(rescoreContext.getWindowSize(), topN);
                        }
                    }
                    // In the QueryPhase we don't need this protection, because it is build into the IndexSearcher,
                    // but here we create collectors ourselves and we need prevent OOM because of crazy an offset and size.
                    topN = Math.min(topN, subSearchContext.searcher().getIndexReader().maxDoc());
                    if (sort == null) {
                        TopScoreDocCollector topScoreDocCollector = new TopScoreDocCollectorManager(topN, null, Integer.MAX_VALUE)
                            .newCollector();
                        collectors = new Collectors(topScoreDocCollector, null);
                    } else {
                        // TODO: can we pass trackTotalHits=subSearchContext.trackTotalHits(){
                        // Note that this would require to catch CollectionTerminatedException
                        collectors = new Collectors(
                            new TopFieldCollectorManager(sort.sort, topN, null, Integer.MAX_VALUE).newCollector(),
                            subSearchContext.trackScores() ? new MaxScoreCollector() : null
                        );
                    }
                    topDocsCollectors.put(bucket, collectors);
                }

                LeafCollector leafCollector = leafCollectors.get(bucket);
                if (leafCollector == null) {
                    leafCollector = collectors.collector.getLeafCollector(aggCtx.getLeafReaderContext());
                    if (scorer != null) {
                        leafCollector.setScorer(scorer);
                    }
                    leafCollectors.put(bucket, leafCollector);
                }
                leafCollector.collect(docId);
            }
        };
    }

    @Override
    public InternalAggregation buildAggregation(long owningBucketOrdinal) throws IOException {
        Collectors collectors = topDocsCollectors.get(owningBucketOrdinal);
        if (collectors == null) {
            return buildEmptyAggregation();
        }
        TopDocsCollector<?> topDocsCollector = collectors.topDocsCollector;
        TopDocs topDocs = topDocsCollector.topDocs();
        float maxScore = Float.NaN;
        if (subSearchContext.sort() == null) {
            for (RescoreContext ctx : subSearchContext.rescore()) {
                try {
                    topDocs = ctx.rescorer().rescore(topDocs, searcher(), ctx);
                } catch (IOException e) {
                    throw new ElasticsearchException("Rescore TopHits Failed", e);
                }
            }
            if (topDocs.scoreDocs.length > 0) {
                maxScore = topDocs.scoreDocs[0].score;
            }
        } else if (subSearchContext.trackScores()) {
            TopFieldCollector.populateScores(topDocs.scoreDocs, subSearchContext.searcher(), subSearchContext.query());
            maxScore = collectors.maxScoreCollector.getMaxScore();
        }
        final TopDocsAndMaxScore topDocsAndMaxScore = new TopDocsAndMaxScore(topDocs, maxScore);
        subSearchContext.queryResult()
            .topDocs(topDocsAndMaxScore, subSearchContext.sort() == null ? null : subSearchContext.sort().formats);
        int[] docIdsToLoad = new int[topDocs.scoreDocs.length];
        for (int i = 0; i < topDocs.scoreDocs.length; i++) {
            docIdsToLoad[i] = topDocs.scoreDocs[i].doc;
        }
        FetchSearchResult fetchResult = runFetchPhase(subSearchContext, docIdsToLoad);
        if (fetchProfiles != null) {
            fetchProfiles.add(fetchResult.profileResult());
        }
        SearchHit[] internalHits = fetchResult.fetchResult().hits().getHits();
        for (int i = 0; i < internalHits.length; i++) {
            ScoreDoc scoreDoc = topDocs.scoreDocs[i];
            SearchHit searchHitFields = internalHits[i];
            searchHitFields.shard(subSearchContext.shardTarget());
            searchHitFields.score(scoreDoc.score);
            if (scoreDoc instanceof FieldDoc fieldDoc) {
                searchHitFields.sortValues(fieldDoc.fields, subSearchContext.sort().formats);
            }
        }
        return new InternalTopHits(
            name,
            subSearchContext.from(),
            subSearchContext.size(),
            topDocsAndMaxScore,
            fetchResult.hits(),
            metadata()
        );
    }

    private static FetchSearchResult runFetchPhase(SubSearchContext subSearchContext, int[] docIdsToLoad) {
        // Fork the search execution context for each slice, because the fetch phase does not support concurrent execution yet.
        SearchExecutionContext searchExecutionContext = new SearchExecutionContext(subSearchContext.getSearchExecutionContext());
        // InnerHitSubContext is not thread-safe, so we fork it as well to support concurrent execution
        InnerHitsContext innerHitsContext = new InnerHitsContext(
            getForkedInnerHits(subSearchContext.innerHits().getInnerHits(), searchExecutionContext)
        );

        SubSearchContext fetchSubSearchContext = new SubSearchContext(subSearchContext) {
            @Override
            public SearchExecutionContext getSearchExecutionContext() {
                return searchExecutionContext;
            }

            @Override
            public InnerHitsContext innerHits() {
                return innerHitsContext;
            }
        };

        fetchSubSearchContext.fetchPhase().execute(fetchSubSearchContext, docIdsToLoad, null);
        return fetchSubSearchContext.fetchResult();
    }

    private static Map<String, InnerHitSubContext> getForkedInnerHits(
        Map<String, InnerHitSubContext> originalInnerHits,
        SearchExecutionContext searchExecutionContext
    ) {
        Map<String, InnerHitSubContext> forkedInnerHits = new HashMap<>();
        for (Map.Entry<String, InnerHitSubContext> entry : originalInnerHits.entrySet()) {
            var forkedContext = entry.getValue().copyWithSearchExecutionContext(searchExecutionContext);
            forkedInnerHits.put(entry.getKey(), forkedContext);
        }

        return forkedInnerHits;
    }

    @Override
    public InternalTopHits buildEmptyAggregation() {
        TopDocs topDocs;
        if (subSearchContext.sort() != null) {
            topDocs = new TopFieldDocs(Lucene.TOTAL_HITS_EQUAL_TO_ZERO, new FieldDoc[0], subSearchContext.sort().sort.getSort());
        } else {
            topDocs = Lucene.EMPTY_TOP_DOCS;
        }
        return new InternalTopHits(
            name,
            subSearchContext.from(),
            subSearchContext.size(),
            new TopDocsAndMaxScore(topDocs, Float.NaN),
            SearchHits.EMPTY_WITH_TOTAL_HITS,
            metadata()
        );
    }

    @Override
    public void collectDebugInfo(BiConsumer<String, Object> add) {
        super.collectDebugInfo(add);
        List<Map<String, Object>> debug = new ArrayList<>();
        for (ProfileResult result : fetchProfiles) {
            Map<String, Object> resultDebug = new HashMap<>();
            resultDebug.put("time", result.getTime());
            resultDebug.put("breakdown", result.getTimeBreakdown());
            debug.add(resultDebug);
        }
        add.accept("fetch_profile", debug);
    }

    @Override
    protected void doClose() {
        Releasables.close(topDocsCollectors, leafCollectors);
    }

    private static class ResettableScorable extends Scorable {

        private Scorable scorable;

        private ResettableScorable(Scorable scorable) {
            this.scorable = scorable;
        }

        private void reset(Scorable scorable) {
            this.scorable = scorable;
        }

        @Override
        public float score() throws IOException {
            return scorable.score();
        }

        @Override
        public float smoothingScore(int docId) throws IOException {
            return scorable.smoothingScore(docId);
        }

        @Override
        public void setMinCompetitiveScore(float minScore) throws IOException {
            scorable.setMinCompetitiveScore(minScore);
        }

        @Override
        public Collection<ChildScorable> getChildren() throws IOException {
            return scorable.getChildren();
        }
    }
}
