"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GroupStream = void 0;
var _lodash = require("lodash");
var _streamsSchema = require("@kbn/streams-schema");
var _esErrors = require("@kbn/es-errors");
var _status_error = require("../../errors/status_error");
var _stream_active_record = require("../stream_active_record/stream_active_record");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class GroupStream extends _stream_active_record.StreamActiveRecord {
  constructor(definition, dependencies) {
    super(definition, dependencies);
  }
  doClone() {
    return new GroupStream((0, _lodash.cloneDeep)(this._definition), this.dependencies);
  }
  async doHandleUpsertChange(definition, desiredState, startingState) {
    if (definition.name !== this._definition.name) {
      return {
        changeStatus: this.changeStatus,
        cascadingChanges: []
      };
    }
    if (!_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      throw new _status_error.StatusError('Cannot change stream types', 400);
    }
    this._definition = definition;
    const missingMembers = [];
    for (const member of this._definition.group.members) {
      if (!desiredState.has(member)) {
        missingMembers.push(member);
      }
    }
    const existsAsDataStream = await Promise.all(missingMembers.map(async member => {
      try {
        const dataStreamResult = await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
          name: member
        });
        return dataStreamResult.data_streams.length > 0 ? member : null;
      } catch (error) {
        if (!(0, _esErrors.isNotFoundError)(error)) {
          throw error;
        }
        return null;
      }
    }));
    const cascadingChanges = existsAsDataStream.filter(member => member !== null).map(member => ({
      type: 'upsert',
      definition: {
        name: member,
        description: '',
        ingest: {
          classic: {},
          lifecycle: {
            inherit: {}
          },
          processing: {
            steps: []
          },
          settings: {}
        }
      }
    }));
    return {
      cascadingChanges,
      changeStatus: 'upserted'
    };
  }
  async doHandleDeleteChange(target, desiredState, startingState) {
    if (target === this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: 'deleted'
      };
    }
    if (!this.isDeleted() && this._definition.group.members.includes(target)) {
      // We need to run validation to check that all related streams still exist
      return {
        cascadingChanges: [],
        changeStatus: 'upserted'
      };
    }
    return {
      cascadingChanges: [],
      changeStatus: this.changeStatus
    };
  }
  async doValidateUpsertion(desiredState, startingState) {
    if (this._definition.name.startsWith('logs.')) {
      throw new _status_error.StatusError('A Group stream name can not start with [logs.]', 400);
    }
    const existsInStartingState = startingState.has(this._definition.name);
    if (!existsInStartingState) {
      // Check for conflicts
      try {
        const dataStreamResult = await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
          name: this._definition.name
        });
        if (dataStreamResult.data_streams.length === 0) {
          return {
            isValid: false,
            errors: [new Error(`Cannot create Group stream ${this.definition.name} due to existing index`)]
          };
        }
        return {
          isValid: false,
          errors: [new Error(`Cannot create Group stream ${this.definition.name} due to existing data stream`)]
        };
      } catch (error) {
        if (!(0, _esErrors.isNotFoundError)(error)) {
          throw error;
        }
      }
    }

    // validate members
    const selfReference = this._definition.group.members.includes(this.name);
    if (selfReference) {
      return {
        isValid: false,
        errors: [new Error(`Group stream ${this.name} cannot have itself as a member`)]
      };
    }
    const missingMembers = [];
    for (const member of this._definition.group.members) {
      const relatedStream = desiredState.get(member);
      if (!relatedStream || relatedStream.isDeleted()) {
        missingMembers.push(member);
      }
    }
    if (missingMembers.length > 0) {
      return {
        isValid: false,
        errors: [new Error(`Group stream ${this.name} has the following members which were not found: ${missingMembers.join(', ')}`)]
      };
    }
    const duplicates = this._definition.group.members.filter((name, index) => this._definition.group.members.indexOf(name) !== index);
    if (duplicates.length !== 0) {
      return {
        isValid: false,
        errors: [new Error(`Group stream ${this.name} cannot have the same member mentioned more than once: ${duplicates.join(',')}`)]
      };
    }
    return {
      isValid: true,
      errors: []
    };
  }
  async doValidateDeletion(desiredState, startingState) {
    const dependentGroupStreams = desiredState.all().filter(stream => stream.name !== this.name && _streamsSchema.Streams.GroupStream.Definition.is(stream.definition) && !stream.isDeleted()).filter(stream => stream.definition.group.members.includes(this.name));
    if (dependentGroupStreams.length !== 0) {
      return {
        isValid: false,
        errors: dependentGroupStreams.map(stream => new Error(`Cannot delete Group stream ${this.name} because Group stream ${stream.name} depends on it`))
      };
    }
    return {
      isValid: true,
      errors: []
    };
  }
  async doDetermineCreateActions() {
    return [{
      type: 'upsert_dot_streams_document',
      request: this._definition
    }];
  }
  async doDetermineUpdateActions() {
    return [{
      type: 'upsert_dot_streams_document',
      request: this._definition
    }];
  }
  async doDetermineDeleteActions() {
    return [{
      type: 'delete_dot_streams_document',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'unlink_assets',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'unlink_features',
      request: {
        name: this._definition.name
      }
    }];
  }
}
exports.GroupStream = GroupStream;