"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetClient = void 0;
exports.getAssetLinkUuid = getAssetLinkUuid;
var _lodash = require("lodash");
var _objectHash = _interopRequireDefault(require("object-hash"));
var _pLimit = _interopRequireDefault(require("p-limit"));
var _assets = require("../../../../common/assets");
var _fields = require("./fields");
var _asset_not_found_error = require("../errors/asset_not_found_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function termQuery(field, value, opts = {
  queryEmptyString: true
}) {
  if (value === null || value === undefined || !opts.queryEmptyString && value === '') {
    return [];
  }
  return [{
    term: {
      [field]: value
    }
  }];
}
function termsQuery(field, values) {
  if (values === null || values === undefined || values.length === 0) {
    return [];
  }
  const filteredValues = values.filter(value => value !== undefined);
  return [{
    terms: {
      [field]: filteredValues
    }
  }];
}
function getAssetLinkUuid(name, asset) {
  return (0, _objectHash.default)({
    [_fields.STREAM_NAME]: name,
    [_fields.ASSET_ID]: asset[_fields.ASSET_ID],
    [_fields.ASSET_TYPE]: asset[_fields.ASSET_TYPE]
  });
}
function toAssetLink(name, asset) {
  return {
    ...asset,
    [_fields.ASSET_UUID]: getAssetLinkUuid(name, asset)
  };
}
function sloSavedObjectToAsset(sloId, savedObject) {
  return {
    [_fields.ASSET_ID]: sloId,
    [_fields.ASSET_TYPE]: 'slo',
    title: savedObject.attributes.name,
    tags: savedObject.attributes.tags.concat(savedObject.references.filter(ref => ref.type === 'tag').map(ref => ref.id))
  };
}
function dashboardSavedObjectToAsset(dashboardId, savedObject) {
  return {
    [_fields.ASSET_ID]: dashboardId,
    [_fields.ASSET_TYPE]: 'dashboard',
    title: savedObject.attributes.title,
    tags: savedObject.references.filter(ref => ref.type === 'tag').map(ref => ref.id)
  };
}
function ruleToAsset(ruleId, rule) {
  return {
    [_fields.ASSET_TYPE]: 'rule',
    [_fields.ASSET_ID]: ruleId,
    title: rule.name,
    tags: rule.tags
  };
}
function fromStorage(link) {
  if (link[_fields.ASSET_TYPE] === 'query') {
    const storedQueryLink = link;
    return {
      ...storedQueryLink,
      query: {
        id: storedQueryLink[_fields.ASSET_ID],
        title: storedQueryLink[_fields.QUERY_TITLE],
        kql: {
          query: storedQueryLink[_fields.QUERY_KQL_BODY]
        },
        feature: storedQueryLink[_fields.QUERY_FEATURE_NAME] ? {
          name: storedQueryLink[_fields.QUERY_FEATURE_NAME],
          filter: JSON.parse(storedQueryLink[_fields.QUERY_FEATURE_FILTER])
        } : undefined
      }
    };
  }
  return link;
}
function toStorage(name, request) {
  const link = toAssetLink(name, request);
  if (link[_fields.ASSET_TYPE] === 'query') {
    const {
      query,
      ...rest
    } = link;
    return {
      ...rest,
      [_fields.STREAM_NAME]: name,
      [_fields.QUERY_TITLE]: query.title,
      [_fields.QUERY_KQL_BODY]: query.kql.query,
      [_fields.QUERY_FEATURE_NAME]: query.feature ? query.feature.name : '',
      [_fields.QUERY_FEATURE_FILTER]: query.feature ? JSON.stringify(query.feature.filter) : ''
    };
  }
  return {
    ...link,
    [_fields.STREAM_NAME]: name
  };
}
class AssetClient {
  constructor(clients) {
    this.clients = clients;
  }
  async linkAsset(name, link) {
    const document = toStorage(name, link);
    await this.clients.storageClient.index({
      id: document[_fields.ASSET_UUID],
      document
    });
    return toAssetLink(name, link);
  }
  async syncAssetList(name, links, assetType) {
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: [...termQuery(_fields.STREAM_NAME, name), ...termQuery(_fields.ASSET_TYPE, assetType)]
        }
      }
    });
    const existingAssetLinks = assetsResponse.hits.hits.map(hit => {
      return fromStorage(hit._source);
    });
    const nextAssetLinks = links.map(link => {
      return toAssetLink(name, link);
    });
    const nextIds = new Set(nextAssetLinks.map(link => link[_fields.ASSET_UUID]));
    const assetLinksDeleted = existingAssetLinks.filter(link => !nextIds.has(link[_fields.ASSET_UUID]));
    const operations = [...assetLinksDeleted.map(asset => ({
      delete: {
        asset
      }
    })), ...nextAssetLinks.map(asset => ({
      index: {
        asset
      }
    }))];
    if (operations.length) {
      await this.bulk(name, operations);
    }
    return {
      deleted: assetLinksDeleted,
      indexed: nextAssetLinks
    };
  }
  async unlinkAsset(name, asset) {
    const id = getAssetLinkUuid(name, asset);
    const {
      result
    } = await this.clients.storageClient.delete({
      id
    });
    if (result === 'not_found') {
      throw new _asset_not_found_error.AssetNotFoundError(`${asset[_fields.ASSET_TYPE]} not found`);
    }
  }
  async clean() {
    await this.clients.storageClient.clean();
  }
  async getAssetLinks(names, assetTypes) {
    const filters = [...termsQuery(_fields.STREAM_NAME, names)];
    if (assetTypes !== null && assetTypes !== void 0 && assetTypes.length) {
      filters.push(...termsQuery(_fields.ASSET_TYPE, assetTypes));
    }
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: filters
        }
      }
    });
    const assetsPerName = names.reduce((acc, name) => {
      acc[name] = [];
      return acc;
    }, {});
    assetsResponse.hits.hits.forEach(hit => {
      const name = hit._source[_fields.STREAM_NAME];
      const asset = fromStorage(hit._source);
      assetsPerName[name].push(asset);
    });
    return assetsPerName;
  }
  async bulkGetByIds(name, assetType, ids) {
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: [...termQuery(_fields.STREAM_NAME, name), ...termQuery(_fields.ASSET_TYPE, assetType), ...termsQuery('_id', ids.map(id => getAssetLinkUuid(name, {
            [_fields.ASSET_TYPE]: assetType,
            [_fields.ASSET_ID]: id
          })))]
        }
      }
    });
    return assetsResponse.hits.hits.map(hit => fromStorage(hit._source));
  }
  async bulk(name, operations) {
    return await this.clients.storageClient.bulk({
      operations: operations.map(operation => {
        if ('index' in operation) {
          const document = toStorage(name, Object.values(operation)[0].asset);
          return {
            index: {
              document,
              _id: document[_fields.ASSET_UUID]
            }
          };
        }
        const id = getAssetLinkUuid(name, operation.delete.asset);
        return {
          delete: {
            _id: id
          }
        };
      }),
      throwOnFail: true
    });
  }
  async getAssets(name) {
    const {
      [name]: assetLinks
    } = await this.getAssetLinks([name]);
    if (assetLinks.length === 0) {
      return [];
    }
    const [queryAssetLinks, savedObjectAssetLinks] = (0, _lodash.partition)(assetLinks, link => link[_fields.ASSET_TYPE] === 'query');
    const idsByType = Object.fromEntries(Object.values(_assets.ASSET_TYPES).map(type => [type, []]));
    savedObjectAssetLinks.forEach(assetLink => {
      const assetType = assetLink['asset.type'];
      const assetId = assetLink['asset.id'];
      idsByType[assetType].push(assetId);
    });
    const limiter = (0, _pLimit.default)(10);
    const [dashboards, rules, slos] = await Promise.all([idsByType.dashboard.length ? this.clients.soClient.bulkGet(idsByType.dashboard.map(dashboardId => ({
      type: 'dashboard',
      id: dashboardId
    }))).then(response => {
      const dashboardsById = (0, _lodash.keyBy)(response.saved_objects, 'id');
      return idsByType.dashboard.flatMap(dashboardId => {
        const dashboard = dashboardsById[dashboardId];
        if (dashboard && !dashboard.error) {
          return [dashboardSavedObjectToAsset(dashboardId, dashboard)];
        }
        return [];
      });
    }) : [], Promise.all(idsByType.rule.map(ruleId => {
      return limiter(() => this.clients.rulesClient.get({
        id: ruleId
      }).then(rule => {
        return ruleToAsset(ruleId, rule);
      }).catch(error => {
        // Handle missing rules gracefully (404, etc.)
        // Return null for missing/inaccessible rules, filter them out later
        return null;
      }));
    })).then(results => results.filter(rule => rule !== null)), idsByType.slo.length ? this.clients.soClient.find({
      type: 'slo',
      filter: `slo.attributes.id:(${idsByType.slo.map(sloId => `"${sloId}"`).join(' OR ')})`,
      perPage: idsByType.slo.length
    }).then(soResponse => {
      const sloDefinitionsById = (0, _lodash.keyBy)(soResponse.saved_objects, 'slo.attributes.id');
      return idsByType.slo.flatMap(sloId => {
        const sloDefinition = sloDefinitionsById[sloId];
        if (sloDefinition && !sloDefinition.error) {
          return [sloSavedObjectToAsset(sloId, sloDefinition)];
        }
        return [];
      });
    }) : []]);
    const savedObjectAssetsWithUuids = [...dashboards, ...rules, ...slos].map(asset => {
      return {
        ...asset,
        [_fields.ASSET_UUID]: getAssetLinkUuid(name, asset)
      };
    });
    return [...savedObjectAssetsWithUuids, ...queryAssetLinks.map(link => {
      return {
        [_fields.ASSET_ID]: link[_fields.ASSET_ID],
        [_fields.ASSET_UUID]: link[_fields.ASSET_UUID],
        [_fields.ASSET_TYPE]: link[_fields.ASSET_TYPE],
        query: link.query,
        title: link.query.title
      };
    })];
  }
  async getSuggestions({
    query,
    assetTypes,
    tags
  }) {
    const perPage = 101;
    const searchAll = !assetTypes;
    const searchDashboardsOrSlos = searchAll || assetTypes.includes('dashboard') || assetTypes.includes('slo');
    const searchRules = searchAll || assetTypes.includes('rule');
    const [suggestionsFromSlosAndDashboards, suggestionsFromRules] = await Promise.all([searchDashboardsOrSlos ? this.clients.soClient.find({
      type: ['dashboard', 'slo'].filter(type => searchAll || assetTypes.includes(type)),
      search: query,
      perPage,
      ...(tags ? {
        hasReferenceOperator: 'OR',
        hasReference: tags.map(tag => ({
          type: 'tag',
          id: tag
        }))
      } : {})
    }).then(results => {
      return results.saved_objects.map(savedObject => {
        if (savedObject.type === 'slo') {
          const sloSavedObject = savedObject;
          return sloSavedObjectToAsset(sloSavedObject.attributes.id, sloSavedObject);
        }
        const dashboardSavedObject = savedObject;
        return dashboardSavedObjectToAsset(dashboardSavedObject.id, dashboardSavedObject);
      });
    }) : Promise.resolve([]), searchRules ? this.clients.rulesClient.find({
      options: {
        perPage,
        ...(tags ? {
          hasReferenceOperator: 'OR',
          hasReference: tags.map(tag => ({
            type: 'tag',
            id: tag
          }))
        } : {})
      }
    }).then(results => {
      return results.data.map(rule => {
        return ruleToAsset(rule.id, rule);
      });
    }) : Promise.resolve([])]);
    return {
      assets: [...suggestionsFromRules, ...suggestionsFromSlosAndDashboards],
      hasMore: Math.max(suggestionsFromSlosAndDashboards.length, suggestionsFromRules.length) > perPage - 1
    };
  }
}
exports.AssetClient = AssetClient;