"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterQueries = filterQueries;
exports.filterRouting = filterRouting;
exports.getFields = getFields;
exports.includedObjectsFor = includedObjectsFor;
exports.scopeContentPackStreams = scopeContentPackStreams;
exports.scopeIncludedObjects = scopeIncludedObjects;
exports.withRootPrefix = withRootPrefix;
exports.withoutBaseFields = withoutBaseFields;
exports.withoutRootPrefix = withoutRootPrefix;
var _lodash = require("lodash");
var _contentPacksSchema = require("@kbn/content-packs-schema");
var _error = require("../error");
var _logs_layer = require("../../streams/component_templates/logs_layer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function withoutRootPrefix(root, name) {
  const prefix = `${root}.`;
  return name.startsWith(prefix) ? name.slice(prefix.length) : name;
}
function withRootPrefix(root, name) {
  return `${root}.${name}`;
}
function includedObjectsFor(stream, include) {
  if ((0, _contentPacksSchema.isIncludeAll)(include)) {
    return include;
  }
  for (const routing of include.objects.routing) {
    if (stream === routing.destination) {
      return routing;
    }
  }
  throw new _error.ContentPackIncludeError(`Could not find included objects for stream [${stream}]`);
}
function filterQueries(entry, include) {
  if ((0, _contentPacksSchema.isIncludeAll)(include)) {
    return entry.request.queries;
  }
  return include.objects.queries.map(({
    id
  }) => {
    const existingQuery = entry.request.queries.find(query => query.id === id);
    if (!existingQuery) {
      throw new _error.ContentPackIncludeError(`Stream [${entry.name}] does not define query [${id}]`);
    }
    return existingQuery;
  });
}
function filterRouting(entry, include) {
  const routing = entry.request.stream.ingest.wired.routing;
  if ((0, _contentPacksSchema.isIncludeAll)(include)) {
    return routing;
  }
  const existingDestinations = new Set(routing.map(({
    destination
  }) => destination));
  include.objects.routing.forEach(({
    destination
  }) => {
    if (!existingDestinations.has(destination)) {
      throw new _error.ContentPackIncludeError(`Stream [${entry.name}] does not route to [${destination}]`);
    }
  });
  return (0, _lodash.intersectionBy)(routing, include.objects.routing, ({
    destination
  }) => destination);
}
function getFields(entry, include) {
  if ((0, _contentPacksSchema.isIncludeAll)(include) || include.objects.mappings) {
    return entry.request.stream.ingest.wired.fields;
  }
  return {};
}
function withoutBaseFields(fields) {
  return Object.keys(fields).filter(key => !_logs_layer.baseFields[key]).reduce((filtered, key) => {
    filtered[key] = (0, _lodash.omit)(fields[key], 'from');
    return filtered;
  }, {});
}
function scopeContentPackStreams({
  root,
  streams
}) {
  return streams.map(stream => ({
    ...stream,
    name: stream.name === _contentPacksSchema.ROOT_STREAM_ID ? root : withRootPrefix(root, stream.name),
    request: {
      ...stream.request,
      stream: {
        ...stream.request.stream,
        ingest: {
          ...stream.request.stream.ingest,
          wired: {
            ...stream.request.stream.ingest.wired,
            routing: stream.request.stream.ingest.wired.routing.map(definition => ({
              ...definition,
              destination: withRootPrefix(root, definition.destination)
            }))
          }
        }
      }
    }
  }));
}
function scopeIncludedObjects({
  root,
  include
}) {
  if ((0, _contentPacksSchema.isIncludeAll)(include)) {
    return include;
  }
  return {
    objects: {
      ...include.objects,
      routing: include.objects.routing.map(routing => ({
        ...scopeIncludedObjects({
          root,
          include: routing
        }),
        destination: withRootPrefix(root, routing.destination)
      }))
    }
  };
}