"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.streamRoutingMachine = exports.createStreamRoutingMachineImplementations = void 0;
var _xstate = require("xstate5");
var _xstateUtils = require("@kbn/xstate-utils");
var _streamsSchema = require("@kbn/streams-schema");
var _streamlang = require("@kbn/streamlang");
var _stream_actors = require("./stream_actors");
var _utils = require("../../utils");
var _selectors = require("./selectors");
var _routing_samples_state_machine = require("./routing_samples_state_machine");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const streamRoutingMachine = exports.streamRoutingMachine = (0, _xstate.setup)({
  types: {
    input: {},
    context: {},
    events: {}
  },
  actors: {
    deleteStream: (0, _xstateUtils.getPlaceholderFor)(_stream_actors.createDeleteStreamActor),
    forkStream: (0, _xstateUtils.getPlaceholderFor)(_stream_actors.createForkStreamActor),
    upsertStream: (0, _xstateUtils.getPlaceholderFor)(_stream_actors.createUpsertStreamActor),
    routingSamplesMachine: (0, _xstateUtils.getPlaceholderFor)(() => _routing_samples_state_machine.routingSamplesMachine)
  },
  actions: {
    notifyStreamSuccess: (0, _xstateUtils.getPlaceholderFor)(_stream_actors.createStreamSuccessNofitier),
    notifyStreamFailure: (0, _xstateUtils.getPlaceholderFor)(_stream_actors.createStreamFailureNofitier),
    refreshDefinition: () => {},
    addNewRoutingRule: (0, _xstate.assign)(({
      context
    }) => {
      const newRule = _utils.routingConverter.toUIDefinition({
        destination: `${context.definition.stream.name}.child`,
        where: _streamlang.ALWAYS_CONDITION,
        status: 'enabled',
        isNew: true
      });
      return {
        currentRuleId: newRule.id,
        routing: [...context.routing, newRule]
      };
    }),
    appendRoutingRules: (0, _xstate.assign)(({
      context
    }, params) => {
      return {
        routing: [...context.routing, ...params.definitions.map(_utils.routingConverter.toUIDefinition)]
      };
    }),
    patchRule: (0, _xstate.assign)(({
      context
    }, params) => ({
      routing: context.routing.map(rule => rule.id === context.currentRuleId ? {
        ...rule,
        ...params.routingRule
      } : rule)
    })),
    reorderRouting: (0, _xstate.assign)((_, params) => ({
      routing: params.routing
    })),
    resetRoutingChanges: (0, _xstate.assign)(({
      context
    }) => ({
      currentRuleId: null,
      routing: context.initialRouting
    })),
    setupRouting: (0, _xstate.assign)((_, params) => {
      const routing = params.definition.stream.ingest.wired.routing.map(_utils.routingConverter.toUIDefinition);
      return {
        currentRuleId: null,
        initialRouting: routing,
        routing
      };
    }),
    storeCurrentRuleId: (0, _xstate.assign)((_, params) => ({
      currentRuleId: params.id
    })),
    storeDefinition: (0, _xstate.assign)((_, params) => ({
      definition: params.definition
    })),
    storeSuggestedRuleId: (0, _xstate.assign)((_, params) => ({
      suggestedRuleId: params.id
    })),
    resetSuggestedRuleId: (0, _xstate.assign)(() => ({
      suggestedRuleId: null
    }))
  },
  guards: {
    canForkStream: (0, _xstate.and)(['hasManagePrivileges', 'isValidRouting', 'isValidChild']),
    canReorderRules: (0, _xstate.and)(['hasManagePrivileges', 'hasMultipleRoutingRules']),
    canUpdateStream: (0, _xstate.and)(['hasManagePrivileges', 'isValidRouting']),
    hasMultipleRoutingRules: ({
      context
    }) => context.routing.length > 1,
    hasManagePrivileges: ({
      context
    }) => context.definition.privileges.manage,
    hasSimulatePrivileges: ({
      context
    }) => context.definition.privileges.simulate,
    isAlreadyEditing: ({
      context
    }, params) => context.currentRuleId === params.id,
    isValidRouting: ({
      context
    }) => (0, _streamsSchema.isSchema)(_streamsSchema.routingDefinitionListSchema, context.routing.map(_utils.routingConverter.toAPIDefinition)),
    isValidChild: ({
      context
    }) => {
      // If there's no current rule, skip validation (e.g., when editing suggestions)
      if (!context.currentRuleId) return true;
      const currentRule = (0, _selectors.selectCurrentRule)(context);
      const currentStream = context.definition.stream;
      return (0, _streamsSchema.isChildOf)(currentStream.name, currentRule.destination);
    }
  }
}).createMachine({
  /** @xstate-layout N4IgpgJg5mDOIC5QCcD2BXALgSwHZQGVNkwBDAWwDo9sdSAbbALzygGIBtABgF1FQADqli1sqXPxAAPRACZZATkoA2AIwAWABwL1sgKybZG1bIA0IAJ6IAzKr2V1XLgs3WA7Jq7uFb5QF8-czQsViISCkpwiAs2WGIyKhIAYzBsADdIbj4kECERHHFJGQRVBS5ZSh9y8uU9PWtrPTdzKwRZdWVKTWVZQz0FOx7ZNz0AoIwcfDCEyLJo6gh6MDZgyagAJXQlyiTwzDAsyTzRQpzi9V1KPWUFAe1Pa2UevRbEVVLOrmv39W9DdTGIFWoXiESiFgWSxWE1Ym22kFohxyxwKEjONk09lkQ00vgMei4yk0rxKhK4lDcdzc1gGqlqdkBwKmoMScwh2EWyyZGy2YFmqGQEDAyCRgmEJzRoGKAFpVNYuuo3HYdKpCQplE5iZY3rIuOpKIpugomnpdN0AYEgTDmeFWaR5rsyGsAHJgADucL5SQAFqR8KxoSF8J6dn6UvRRblxaiim8vJouho9HLNJjk8p1CSbhUCXTGj51WprIzrYQWbN7RDHaQXe6Qz6-VAA9z6778AdeEdo2JJdI3o8lNZ1Ap3Lj2t0ta0iRVbCPDO9rJ4TCWg2XbRWHXtWK6Pbydm2m-hA2sQwjMJGUT3YyUMzm9L1hh03PpVCTKaoHNcDI0NNiRyu1mmME2R2Ld8B3VtG2bUsQwAMwFABrC9u1OKU41+FRHBpW5qWUIdM21Uk9QpDV500Ewbn-S1uSAu1Nydbc6z3eDkAQgMIHEPk8DSVAEL5FiENogBBJJMAFZD8ivdEEGpD8enKNxFMcZ8BjfYdKBpJpFEJG4mlUACQXXcFQIY8CmO2ASA2FNBkEoAR6BrFiqAE4TRPEztkRQ3tilVHwHGpLRVBGWx3jMQjtHsJofFqU03EVTQLXGVdaI3CEz1hPcG39I8W0yvYO2yMVJNQvsSkaTolQ8Wd3gSkcSVC-UPF1OpFRcPVi2o0sUuM9Lg0yg9oNXeswzACMPKKiVr0MeVXD1dV3D0lx6tNckOgUXUjC+dp5AMm0Zh6iBRD67YssPdhcpOg8Cq7YrvLeNxnEoPUCWxZQ4sVRd6rehMLmfDQuA8e9Rk65LywOo6eUuqCcpgvczwkybpLsIwHFUWbzX0Wd6pChwCwUh7XCaXa132kDeshr0Bphoa4cO89VEKqNbuvIKOieoldB6X5+n0erfiUfouA0dx3jnNxie6sm6YyqHsvO2HthIchUAyBGYyRoLOkcbp3Cedb1HqPn2hUP8Ae2g2EolsGpYhyC5ePGW+VgUhVfGpnEbQkoPgNXodF6AxdEeerfHJVVuiFgHbhMcWQcA63K0ocmQyVlXHbYDjcC43AeL4yghSWfZXLEkU3cvErikaNwunI-n4rUIKXkIkwLie75hcpf4raMm2Tz3FO0jT6yBTshzMCcvPRrAQuWRE4u1akz2TEUfy6gDnRasnN56i12bCXkWRGkMLvSYTpO93QAQIBrNOM6znO+Qv2BhUwIv3MZsu7oQSuDSCw+LnKRUFxg76EoHYU0Pg6RaHXsfYCp9pbHQfpfa+CC2BD1svZRyAoqCP2fq-Eu78vIs3kAmciGhfB-SGF9IcVxGh6QzN8PUMC6IQhIAKIUyBHawH5IKYUg1e6KzAGw4U89y79h6D7awAMCRaCHAoEkzVyRxV-v0OUdJ9Kx0MifeYrCeEcIQVwnR7C+GO1DLgcMIjP4Eh+joY0NIYqmkbq0XUQUnoAIaLoAGHQmGpW4UY-RvjeHU34U7F211PLMyRl8ckTxhw6BuAlRQyh5ENCrpRcoi40Z4UYRovasDtGCN0ZwygF8r5x1tOnTi1Bs68UQU-ZAL8Z5uXwTdD2pVA5XHXs+A+9Rky-HkUYTopR1ClEeIqL43jjKGMCRTLhJTkEkwoKg5ANkR6YOQNggQdSGm2lnm-Fp6tF6qnJLqDwHclRAMIvIR6AwMkJSVO4AIlpcCoCFPAHINEWT7IXqVWUj07nKmGWqDUr4m4t3oWUDwPhXDWHaN4mgdBGAsHwF80RbQ6RdCeEYIKQVPDDmsMkioSYyg3LRnmYGSUylaNaBNA5Py1AKiVP0QFNxgUkkXFXWcjgfAjHvOMnJCzmGQjACiz++glBqjuG1R4zwSQJQTD0P6DR1TkRjhSzReSqxgSgBBXkIrrxKnJF8LQwx9CeF6CSe8H5yIjluPoYZosJkgWrLWXcsszp6qRv7BwQxKTbVxORWVGgHANB+KmLa6i1W5MFc6xirr+KIVYB6z2IwExGsMM+Aw5RN4IG0PKOoeELiY2-AoR1cDba6vCa04oNwq4G3Ws+fM1I5FN01v5bmdqPH+H5ZLMtwT9zQygEm0qdJ5AaV9QYXwBtxzLRHBpGxdR1QE1VVaUG3de0mP7o7IdxQRgVCqvXOVcU9TY1VJUd6e94wDFLfMM+2w5nBO3YgOK9g60msbTSYOjwVBcExPofQ1JTSaGvSwgpfiZmPoQLpCkcVfC-qFsMfFTd5oGmGcaHl7hhyyGAwEvRMycOJsrbSnyki80UUUC4GkQ4kmXLId6-ouIyjOEkeSldlKNX4f8fe9V5AINxXlEpWDvLo6IacScroZR3AwphZSU0jy-BAA */
  id: 'routingStream',
  context: ({
    input
  }) => ({
    currentRuleId: null,
    definition: input.definition,
    initialRouting: [],
    routing: [],
    suggestedRuleId: null
  }),
  initial: 'initializing',
  states: {
    initializing: {
      always: 'ready'
    },
    ready: {
      id: 'ready',
      initial: 'idle',
      entry: [{
        type: 'setupRouting',
        params: ({
          context
        }) => ({
          definition: context.definition
        })
      }],
      on: {
        'stream.received': {
          target: '#ready',
          actions: [{
            type: 'storeDefinition',
            params: ({
              event
            }) => event
          }],
          reenter: true
        },
        'routingSamples.setDocumentMatchFilter': {
          actions: (0, _xstate.enqueueActions)(({
            enqueue,
            event
          }) => {
            enqueue.sendTo('routingSamplesMachine', {
              type: 'routingSamples.setDocumentMatchFilter',
              filter: event.filter
            });
          })
        },
        'suggestion.preview': {
          target: '#idle',
          actions: (0, _xstate.enqueueActions)(({
            enqueue,
            event
          }) => {
            enqueue.sendTo('routingSamplesMachine', {
              type: 'routingSamples.setSelectedPreview',
              preview: event.toggle ? {
                type: 'suggestion',
                name: event.name,
                index: event.index
              } : undefined
            });
            enqueue.sendTo('routingSamplesMachine', {
              type: 'routingSamples.updateCondition',
              condition: event.toggle ? event.condition : undefined
            });
            enqueue.sendTo('routingSamplesMachine', {
              type: 'routingSamples.setDocumentMatchFilter',
              filter: 'matched'
            });
          })
        },
        'routingRule.reviewSuggested': {
          target: '#ready.reviewSuggestedRule',
          actions: [{
            type: 'storeSuggestedRuleId',
            params: ({
              event
            }) => event
          }]
        }
      },
      invoke: {
        id: 'routingSamplesMachine',
        src: 'routingSamplesMachine',
        input: ({
          context
        }) => ({
          definition: context.definition,
          documentMatchFilter: 'matched'
        })
      },
      states: {
        idle: {
          id: 'idle',
          on: {
            'routingRule.create': {
              guard: 'hasSimulatePrivileges',
              target: 'creatingNewRule'
            },
            'routingRule.edit': {
              guard: 'hasManagePrivileges',
              target: 'editingRule',
              actions: [{
                type: 'storeCurrentRuleId',
                params: ({
                  event
                }) => event
              }]
            },
            'routingRule.reorder': {
              guard: 'canReorderRules',
              target: 'reorderingRules',
              actions: [{
                type: 'reorderRouting',
                params: ({
                  event
                }) => event
              }]
            }
          }
        },
        creatingNewRule: {
          id: 'creatingNewRule',
          entry: [{
            type: 'addNewRoutingRule'
          }, (0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.setSelectedPreview',
            preview: {
              type: 'createStream'
            }
          }), (0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.updateCondition',
            condition: {
              always: {}
            }
          })],
          exit: [{
            type: 'resetRoutingChanges'
          }, (0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.setSelectedPreview',
            preview: undefined
          }), (0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.updateCondition',
            condition: undefined
          }), (0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.setDocumentMatchFilter',
            filter: 'matched'
          })],
          initial: 'changing',
          states: {
            changing: {
              on: {
                'routingRule.cancel': {
                  target: '#idle',
                  actions: [{
                    type: 'resetRoutingChanges'
                  }, (0, _xstate.sendTo)('routingSamplesMachine', {
                    type: 'routingSamples.setDocumentMatchFilter',
                    filter: 'matched'
                  })]
                },
                'routingRule.change': {
                  actions: (0, _xstate.enqueueActions)(({
                    enqueue,
                    event
                  }) => {
                    enqueue({
                      type: 'patchRule',
                      params: {
                        routingRule: event.routingRule
                      }
                    });

                    // Trigger samples collection only on condition change
                    if (event.routingRule.where) {
                      enqueue.sendTo('routingSamplesMachine', {
                        type: 'routingSamples.updateCondition',
                        condition: event.routingRule.where
                      });
                    }
                  })
                },
                'routingRule.edit': {
                  guard: 'hasManagePrivileges',
                  target: '#editingRule',
                  actions: [{
                    type: 'storeCurrentRuleId',
                    params: ({
                      event
                    }) => event
                  }]
                },
                'routingRule.fork': {
                  guard: 'canForkStream',
                  target: 'forking'
                }
              }
            },
            forking: {
              invoke: {
                id: 'forkStreamActor',
                src: 'forkStream',
                input: ({
                  context
                }) => {
                  const currentRoutingRule = (0, _selectors.selectCurrentRule)(context);
                  return {
                    definition: context.definition,
                    where: currentRoutingRule.where,
                    destination: currentRoutingRule.destination,
                    status: currentRoutingRule.status
                  };
                },
                onDone: {
                  target: '#idle',
                  actions: [{
                    type: 'refreshDefinition'
                  }]
                },
                onError: {
                  target: 'changing',
                  actions: [{
                    type: 'notifyStreamFailure'
                  }]
                }
              }
            }
          }
        },
        editingRule: {
          id: 'editingRule',
          initial: 'changing',
          entry: [(0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.setSelectedPreview',
            preview: {
              type: 'updateStream'
            }
          })],
          exit: [{
            type: 'resetRoutingChanges'
          }],
          states: {
            changing: {
              on: {
                'routingRule.create': {
                  guard: 'hasSimulatePrivileges',
                  target: '#creatingNewRule'
                },
                'routingRule.cancel': {
                  target: '#idle',
                  actions: [{
                    type: 'resetRoutingChanges'
                  }, (0, _xstate.sendTo)('routingSamplesMachine', {
                    type: 'routingSamples.setDocumentMatchFilter',
                    filter: 'matched'
                  })]
                },
                'routingRule.change': {
                  actions: [{
                    type: 'patchRule',
                    params: ({
                      event
                    }) => event
                  }]
                },
                'routingRule.edit': [{
                  guard: {
                    type: 'isAlreadyEditing',
                    params: ({
                      event
                    }) => event
                  },
                  target: '#idle',
                  actions: [{
                    type: 'storeCurrentRuleId',
                    params: {
                      id: null
                    }
                  }]
                }, {
                  actions: [{
                    type: 'storeCurrentRuleId',
                    params: ({
                      event
                    }) => event
                  }]
                }],
                'routingRule.remove': {
                  guard: 'hasManagePrivileges',
                  target: 'removingRule'
                },
                'routingRule.save': {
                  guard: 'canUpdateStream',
                  target: 'updatingRule'
                }
              }
            },
            removingRule: {
              invoke: {
                id: 'deleteStreamActor',
                src: 'deleteStream',
                input: ({
                  context
                }) => ({
                  name: (0, _selectors.selectCurrentRule)(context).destination
                }),
                onDone: {
                  target: '#idle',
                  actions: [{
                    type: 'refreshDefinition'
                  }]
                },
                onError: {
                  target: 'changing'
                }
              }
            },
            updatingRule: {
              invoke: {
                id: 'upsertStreamActor',
                src: 'upsertStream',
                input: ({
                  context
                }) => ({
                  definition: context.definition,
                  routing: context.routing.map(_utils.routingConverter.toAPIDefinition)
                }),
                onDone: {
                  target: '#idle',
                  actions: [{
                    type: 'notifyStreamSuccess'
                  }, {
                    type: 'refreshDefinition'
                  }]
                },
                onError: {
                  target: 'changing',
                  actions: [{
                    type: 'notifyStreamFailure'
                  }]
                }
              }
            }
          }
        },
        reorderingRules: {
          id: 'reorderingRules',
          initial: 'reordering',
          entry: [(0, _xstate.sendTo)('routingSamplesMachine', {
            type: 'routingSamples.setSelectedPreview',
            preview: {
              type: 'updateStream'
            }
          })],
          states: {
            reordering: {
              on: {
                'routingRule.reorder': {
                  actions: [{
                    type: 'reorderRouting',
                    params: ({
                      event
                    }) => event
                  }]
                },
                'routingRule.cancel': {
                  target: '#idle',
                  actions: [{
                    type: 'resetRoutingChanges'
                  }]
                },
                'routingRule.save': {
                  guard: 'canUpdateStream',
                  target: 'updatingStream'
                }
              }
            },
            updatingStream: {
              invoke: {
                id: 'upsertStreamActor',
                src: 'upsertStream',
                input: ({
                  context
                }) => ({
                  definition: context.definition,
                  routing: context.routing.map(_utils.routingConverter.toAPIDefinition)
                }),
                onDone: {
                  target: '#idle',
                  actions: [{
                    type: 'notifyStreamSuccess'
                  }, {
                    type: 'refreshDefinition'
                  }]
                },
                onError: {
                  target: 'reordering',
                  actions: [{
                    type: 'notifyStreamFailure'
                  }]
                }
              }
            }
          }
        },
        reviewSuggestedRule: {
          id: 'reviewSuggestedRule',
          initial: 'reviewing',
          states: {
            reviewing: {
              on: {
                'routingRule.fork': {
                  guard: 'canForkStream',
                  target: 'forking'
                },
                'routingRule.cancel': {
                  target: '#idle',
                  actions: [{
                    type: 'resetSuggestedRuleId'
                  }]
                }
              }
            },
            forking: {
              invoke: {
                id: 'forkStreamActor',
                src: 'forkStream',
                input: ({
                  context,
                  event
                }) => {
                  (0, _xstate.assertEvent)(event, 'routingRule.fork');
                  const {
                    routingRule
                  } = event;
                  if (!routingRule) {
                    throw new Error('No routing rule to fork');
                  }
                  return {
                    definition: context.definition,
                    destination: routingRule.destination,
                    where: routingRule.where,
                    status: 'enabled'
                  };
                },
                onDone: {
                  target: '#idle',
                  actions: [{
                    type: 'refreshDefinition'
                  }, {
                    type: 'resetSuggestedRuleId'
                  }]
                },
                onError: {
                  target: 'reviewing',
                  actions: [{
                    type: 'notifyStreamFailure'
                  }]
                }
              }
            }
          }
        }
      }
    }
  }
});
const createStreamRoutingMachineImplementations = ({
  refreshDefinition,
  streamsRepositoryClient,
  core,
  data,
  timeState$,
  forkSuccessNofitier,
  telemetryClient
}) => ({
  actors: {
    deleteStream: (0, _stream_actors.createDeleteStreamActor)({
      streamsRepositoryClient
    }),
    forkStream: (0, _stream_actors.createForkStreamActor)({
      streamsRepositoryClient,
      forkSuccessNofitier,
      telemetryClient
    }),
    upsertStream: (0, _stream_actors.createUpsertStreamActor)({
      streamsRepositoryClient
    }),
    routingSamplesMachine: _routing_samples_state_machine.routingSamplesMachine.provide((0, _routing_samples_state_machine.createRoutingSamplesMachineImplementations)({
      data,
      timeState$
    }))
  },
  actions: {
    refreshDefinition,
    notifyStreamSuccess: (0, _stream_actors.createStreamSuccessNofitier)({
      toasts: core.notifications.toasts
    }),
    notifyStreamFailure: (0, _stream_actors.createStreamFailureNofitier)({
      toasts: core.notifications.toasts
    })
  }
});
exports.createStreamRoutingMachineImplementations = createStreamRoutingMachineImplementations;