"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDeleteStreamActor = createDeleteStreamActor;
exports.createForkStreamActor = createForkStreamActor;
exports.createStreamSuccessNofitier = exports.createStreamFailureNofitier = void 0;
exports.createUpsertStreamActor = createUpsertStreamActor;
var _xstate = require("xstate5");
var _i18n = require("@kbn/i18n");
var _errors = require("../../../../../util/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Upsert stream actor factory
 * This actor is used to update the routing of a stream
 */

function createUpsertStreamActor({
  streamsRepositoryClient
}) {
  return (0, _xstate.fromPromise)(({
    input,
    signal
  }) => {
    return streamsRepositoryClient.fetch(`PUT /api/streams/{name}/_ingest 2023-10-31`, {
      signal,
      params: {
        path: {
          name: input.definition.stream.name
        },
        body: {
          ingest: {
            ...input.definition.stream.ingest,
            wired: {
              ...input.definition.stream.ingest.wired,
              routing: input.routing
            }
          }
        }
      }
    });
  });
}

/**
 * Fork stream actor factory
 * This actor is used to fork a stream, creating a new stream with the same definition
 */

function createForkStreamActor({
  streamsRepositoryClient,
  forkSuccessNofitier,
  telemetryClient
}) {
  return (0, _xstate.fromPromise)(async ({
    input,
    signal
  }) => {
    const response = await streamsRepositoryClient.fetch('POST /api/streams/{name}/_fork 2023-10-31', {
      signal,
      params: {
        path: {
          name: input.definition.stream.name
        },
        body: {
          where: input.where,
          status: input.status,
          stream: {
            name: input.destination
          }
        }
      }
    });
    forkSuccessNofitier(input.destination);
    telemetryClient.trackChildStreamCreated({
      name: input.destination
    });
    return response;
  });
}

/**
 * Delete stream actor factory
 * This actor is used to fork a stream, creating a new stream with the same definition
 */

function createDeleteStreamActor({
  streamsRepositoryClient
}) {
  return (0, _xstate.fromPromise)(({
    input,
    signal
  }) => {
    return streamsRepositoryClient.fetch('DELETE /api/streams/{name} 2023-10-31', {
      signal,
      params: {
        path: {
          name: input.name
        }
      }
    });
  });
}

/**
 * Notifier factories
 */

const createStreamSuccessNofitier = ({
  toasts
}) => () => {
  toasts.addSuccess({
    title: _i18n.i18n.translate('xpack.streams.streamDetailRouting.saved', {
      defaultMessage: 'Stream saved'
    }),
    toastLifeTimeMs: 3000
  });
};
exports.createStreamSuccessNofitier = createStreamSuccessNofitier;
const createStreamFailureNofitier = ({
  toasts
}) => params => {
  const event = params.event;
  const formattedError = (0, _errors.getFormattedError)(event.error);
  toasts.addError(formattedError, {
    title: _i18n.i18n.translate('xpack.streams.failedToSave', {
      defaultMessage: 'Failed to save'
    }),
    toastMessage: formattedError.message,
    toastLifeTimeMs: 5000
  });
};
exports.createStreamFailureNofitier = createStreamFailureNofitier;