"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResilientConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fp = require("lodash/fp");
var _server = require("@kbn/actions-plugin/server");
var _configSchema = require("@kbn/config-schema");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var i18n = _interopRequireWildcard(require("./translations"));
var _constants = require("./constants");
var _schema = require("./schema");
var _utils = require("./utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const VIEW_INCIDENT_URL = `#incidents`;
class ResilientConnector extends _server.CaseConnector {
  constructor(params, pushToServiceParamsExtendedSchema) {
    super(params, pushToServiceParamsExtendedSchema);
    (0, _defineProperty2.default)(this, "urls", void 0);
    (0, _defineProperty2.default)(this, "getIncidentFieldsUrl", () => `${this.getOrgUrl()}/types/incident/fields`);
    this.urls = {
      incidentTypes: `${this.getIncidentFieldsUrl()}/incident_type_ids`,
      incident: `${this.getOrgUrl()}/incidents`,
      comment: `${this.getOrgUrl()}/incidents/{inc_id}/comments`,
      severity: `${this.getIncidentFieldsUrl()}/severity_code`
    };
    this.registerSubActions();
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response2, _error$response3;
    if (!((_error$response = error.response) !== null && _error$response !== void 0 && _error$response.status)) {
      return i18n.UNKNOWN_API_ERROR;
    }
    if (error.response.status === 401) {
      return i18n.UNAUTHORIZED_API_ERROR;
    }
    if ((0, _fp.isObject)((_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : _error$response2.data) && 'message' in error.response.data) {
      return `API Error: ${error.response.data.message}`;
    }
    return `API Error: ${(_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.statusText}`;
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.SUB_ACTION.INCIDENT_TYPES,
      method: 'getIncidentTypes',
      schema: _schema.ExecutorSubActionGetIncidentTypesParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.SEVERITY,
      method: 'getSeverity',
      schema: _schema.ExecutorSubActionGetSeverityParamsSchema
    });
    this.registerSubAction({
      name: _constants.SUB_ACTION.FIELDS,
      method: 'getFields',
      schema: _schema.ExecutorSubActionCommonFieldsParamsSchema
    });
  }
  getAuthHeaders() {
    return (0, _server.getBasicAuthHeader)({
      username: this.secrets.apiKeyId,
      password: this.secrets.apiKeySecret
    });
  }
  getOrgUrl() {
    const {
      apiUrl: url,
      orgId
    } = this.config;
    return `${url}/rest/orgs/${orgId}`;
  }
  getIncidentViewURL(key) {
    const url = this.config.apiUrl;
    const urlWithoutTrailingSlash = url.endsWith('/') ? url.slice(0, -1) : url;
    return `${urlWithoutTrailingSlash}/${VIEW_INCIDENT_URL}/${key}`;
  }
  async createIncident(incident, connectorUsageCollector) {
    try {
      let data = {
        name: incident.name,
        discovered_date: Date.now()
      };
      if (incident !== null && incident !== void 0 && incident.description) {
        var _incident$description;
        data = {
          ...data,
          description: {
            format: 'html',
            content: (_incident$description = incident.description) !== null && _incident$description !== void 0 ? _incident$description : ''
          }
        };
      }
      if (incident !== null && incident !== void 0 && incident.incidentTypes) {
        data = {
          ...data,
          incident_type_ids: incident.incidentTypes.map(id => ({
            id: Number(id)
          }))
        };
      }
      if (incident !== null && incident !== void 0 && incident.severityCode) {
        data = {
          ...data,
          severity_code: {
            id: Number(incident.severityCode)
          }
        };
      }
      if (incident.additionalFields) {
        const fieldsMetaData = await this.getFields({}, connectorUsageCollector);
        const {
          properties,
          ...rest
        } = (0, _utils.prepareAdditionalFieldsForCreation)(fieldsMetaData, incident.additionalFields);
        data = {
          ...data,
          ...(rest ? rest : {}),
          ...(properties ? {
            properties
          } : {})
        };
      }
      const res = await this.request({
        url: `${this.urls.incident}?text_content_output_format=objects_convert`,
        method: 'POST',
        data,
        headers: this.getAuthHeaders(),
        responseSchema: _configSchema.schema.object({
          id: _configSchema.schema.number(),
          create_date: _configSchema.schema.number()
        }, {
          unknowns: 'allow'
        })
      }, connectorUsageCollector);
      const {
        id,
        create_date: createDate
      } = res.data;
      return {
        title: `${id}`,
        id: `${id}`,
        pushedDate: new Date(createDate).toISOString(),
        url: this.getIncidentViewURL(id.toString())
      };
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to create incident. Error: ${error.message}.`));
    }
  }
  async updateIncident({
    incidentId,
    incident
  }, connectorUsageCollector) {
    try {
      const latestIncident = await this.getIncident({
        id: incidentId
      }, connectorUsageCollector);
      const fields = await this.getFields({}, connectorUsageCollector);

      // Remove null or undefined values. Allowing null values sets the field in IBM Resilient to empty.
      const newIncident = (0, _fp.omitBy)(_fp.isNil, incident);
      const data = (0, _utils.formatUpdateRequest)({
        oldIncident: latestIncident,
        newIncident,
        fields
      });
      const res = await this.request({
        method: 'PATCH',
        url: `${this.urls.incident}/${incidentId}`,
        data,
        headers: this.getAuthHeaders(),
        responseSchema: _configSchema.schema.object({
          success: _configSchema.schema.boolean(),
          message: _configSchema.schema.nullable(_configSchema.schema.string())
        }, {
          unknowns: 'allow'
        })
      }, connectorUsageCollector);
      if (!res.data.success) {
        throw new Error(`Error while updating incident: ${res.data.message}`);
      }
      const updatedIncident = await this.getIncident({
        id: incidentId
      }, connectorUsageCollector);
      return {
        title: `${updatedIncident.id}`,
        id: `${updatedIncident.id}`,
        pushedDate: new Date(updatedIncident.inc_last_modified_date).toISOString(),
        url: this.getIncidentViewURL(updatedIncident.id.toString())
      };
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to update incident with id ${incidentId}. Error: ${error.message}.`));
    }
  }
  async addComment({
    incidentId,
    comment
  }, connectorUsageCollector) {
    try {
      await this.request({
        method: 'POST',
        url: this.urls.comment.replace('{inc_id}', incidentId),
        data: {
          text: {
            format: 'text',
            content: comment
          }
        },
        headers: this.getAuthHeaders(),
        responseSchema: _configSchema.schema.object({}, {
          unknowns: 'allow'
        })
      }, connectorUsageCollector);
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to create comment at incident with id ${incidentId}. Error: ${error.message}.`));
    }
  }
  async getIncident({
    id
  }, connectorUsageCollector) {
    try {
      const res = await this.request({
        method: 'GET',
        url: `${this.urls.incident}/${id}`,
        params: {
          text_content_output_format: 'objects_convert'
        },
        headers: this.getAuthHeaders(),
        responseSchema: _schema.GetIncidentResponseSchema
      }, connectorUsageCollector);
      return res.data;
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to get incident with id ${id}. Error: ${error.message}.`));
    }
  }
  async getIncidentTypes(params, connectorUsageCollector) {
    try {
      var _res$data$values, _res$data;
      const res = await this.request({
        method: 'GET',
        url: this.urls.incidentTypes,
        headers: this.getAuthHeaders(),
        responseSchema: _schema.GetIncidentTypesResponseSchema
      }, connectorUsageCollector);
      const incidentTypes = (_res$data$values = (_res$data = res.data) === null || _res$data === void 0 ? void 0 : _res$data.values) !== null && _res$data$values !== void 0 ? _res$data$values : [];
      return incidentTypes.map(type => ({
        id: type.value.toString(),
        name: type.label
      }));
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to get incident types. Error: ${error.message}.`));
    }
  }
  async getSeverity(params, connectorUsageCollector) {
    try {
      var _res$data$values2, _res$data2;
      const res = await this.request({
        method: 'GET',
        url: this.urls.severity,
        headers: this.getAuthHeaders(),
        responseSchema: _schema.GetSeverityResponseSchema
      }, connectorUsageCollector);
      const severities = (_res$data$values2 = (_res$data2 = res.data) === null || _res$data2 === void 0 ? void 0 : _res$data2.values) !== null && _res$data$values2 !== void 0 ? _res$data$values2 : [];
      return severities.map(type => ({
        id: type.value.toString(),
        name: type.label
      }));
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to get severity. Error: ${error.message}.`));
    }
  }
  async getFields(params, connectorUsageCollector) {
    try {
      const res = await this.request({
        method: 'GET',
        url: this.getIncidentFieldsUrl(),
        headers: this.getAuthHeaders(),
        responseSchema: _schema.GetCommonFieldsResponseSchema
      }, connectorUsageCollector);
      const fields = res.data.map(field => {
        return {
          name: field.name,
          input_type: field.input_type,
          read_only: field.read_only,
          required: field.required,
          text: field.text,
          prefix: field.prefix,
          values: field.values
        };
      });
      return fields;
    } catch (error) {
      throw new Error((0, _axios_utils.getErrorMessage)(i18n.NAME, `Unable to get fields. Error: ${error.message}.`));
    }
  }
}
exports.ResilientConnector = ResilientConnector;