"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.solutionSchema = exports.getSpaceSchema = exports.SPACE_ID_REGEX = void 0;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SPACE_ID_REGEX = exports.SPACE_ID_REGEX = /^[a-z0-9_\-]+$/;
const spaceSchema = _configSchema.schema.object({
  id: _configSchema.schema.string({
    validate: value => {
      if (!SPACE_ID_REGEX.test(value)) {
        return `must be lower case, a-z, 0-9, '_', and '-' are allowed`;
      }
    },
    meta: {
      description: 'The space ID that is part of the Kibana URL when inside the space. Space IDs are limited to lowercase alphanumeric, underscore, and hyphen characters (a-z, 0-9, _, and -). You are cannot change the ID with the update operation.'
    }
  }),
  name: _configSchema.schema.string({
    minLength: 1,
    meta: {
      description: 'The display name for the space. '
    }
  }),
  description: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'A description for the space.'
    }
  })),
  initials: _configSchema.schema.maybe(_configSchema.schema.string({
    maxLength: _common.MAX_SPACE_INITIALS,
    meta: {
      description: 'One or two characters that are shown in the space avatar. By default, the initials are automatically generated from the space name.'
    }
  })),
  color: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: value => {
      if (!/^#[a-zA-Z0-9]{6}$/.test(value)) {
        return `must be a 6 digit hex color, starting with a #`;
      }
    },
    meta: {
      description: 'The hexadecimal color code used in the space avatar. By default, the color is automatically generated from the space name.'
    }
  })),
  disabledFeatures: _configSchema.schema.arrayOf(_configSchema.schema.string({
    meta: {
      description: 'The list of features that are turned off in the space.'
    }
  }), {
    defaultValue: []
  }),
  _reserved: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  imageUrl: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: value => {
      if (value !== '' && !/^data:image.*$/.test(value)) {
        return `must start with 'data:image'`;
      }
    },
    meta: {
      description: 'The data-URL encoded image to display in the space avatar. If specified, initials will not be displayed and the color will be visible as the background color for transparent images. For best results, your image should be 64x64. Images will not be optimized by this API call, so care should be taken when using custom images.'
    }
  }))
});
const solutionSchema = exports.solutionSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('security'), _configSchema.schema.literal('oblt'), _configSchema.schema.literal('es'), _configSchema.schema.literal('classic')]);
const getSpaceSchema = isServerless => {
  if (isServerless) {
    return spaceSchema;
  }
  return spaceSchema.extends({
    solution: _configSchema.schema.maybe(solutionSchema)
  });
};
exports.getSpaceSchema = getSpaceSchema;