"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TourManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TourManager {
  constructor(core, spacesManager) {
    (0, _defineProperty2.default)(this, "tourState$", new _rxjs.BehaviorSubject('not_started'));
    (0, _defineProperty2.default)(this, "showTour$", this.tourState$.pipe((0, _rxjs.map)(state => state === 'in_progress')));
    this.core = core;
    this.spacesManager = spacesManager;
  }
  async startTour() {
    const core = await this.core();
    if (!canManageSpaces(core) || hasCompletedTour(core)) {
      return {
        result: 'not_available'
      };
    }
    if (await hasSingleDefaultSolutionSpace(this.spacesManager)) {
      // we have a single space, and it's not the classic solution, so we can show the tour
      // presumably the user is the admin who created the deployment
      this.tourState$.next('in_progress');
      return {
        result: 'started'
      };
    } else {
      // If we have either (1) multiple space or (2) only one space and it's the default space with the classic solution,
      // we don't want to show the tour later on. This can happen in the following scenarios:
      // - the user deletes all the spaces but one (and that last space has a solution set)
      // - the user edits the default space and sets a solution
      // So we can immediately hide the tour in the global settings from now on.
      await preserveTourCompletion(core);
      return {
        result: 'not_available'
      };
    }
  }
  async finishTour() {
    this.tourState$.next('ended');
    await preserveTourCompletion(await this.core());
  }
  async waitForTourEnd() {
    return (0, _rxjs.firstValueFrom)(this.tourState$.pipe((0, _rxjs.first)(state => state === 'ended'), (0, _rxjs.map)(() => void 0)));
  }
}
exports.TourManager = TourManager;
const canManageSpaces = core => {
  var _core$application$cap;
  return ((_core$application$cap = core.application.capabilities.spaces) === null || _core$application$cap === void 0 ? void 0 : _core$application$cap.manage) === true;
};

// the tour is shown only once for the first admin user, presumably the one who created the deployment
const hasCompletedTour = core => {
  const showTourUiSettingValue = core.settings.globalClient.get(_constants.SHOW_SPACE_SOLUTION_TOUR_SETTING);
  return showTourUiSettingValue === false;
};
const preserveTourCompletion = async core => {
  try {
    await core.settings.globalClient.set(_constants.SHOW_SPACE_SOLUTION_TOUR_SETTING, false);
  } catch (error) {
    // Silently swallow errors, the user will just see the tour again next time they load the page
  }
};
const hasSingleDefaultSolutionSpace = async spacesManager => {
  try {
    const allSpaces = await spacesManager.getSpaces();
    if (allSpaces.length > 1) {
      return false;
    }
    const defaultSpace = allSpaces.find(space => space.id === _constants.DEFAULT_SPACE_ID);
    if (!defaultSpace) {
      return false;
    }
    if (defaultSpace.solution && defaultSpace.solution !== _constants.SOLUTION_VIEW_CLASSIC) {
      return true;
    }
    return false;
  } catch (error) {
    return false;
  }
};