"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CreateSLO = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _uuid = require("uuid");
var _constants2 = require("../../common/constants");
var _sli_pipeline_template = require("../assets/ingest_templates/sli_pipeline_template");
var _summary_pipeline_template = require("../assets/ingest_templates/summary_pipeline_template");
var _models = require("../domain/models");
var _services = require("../domain/services");
var _errors = require("../errors");
var _saved_objects = require("../saved_objects");
var _retry = require("../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
var _assert_expected_indicator_source_index_privileges = require("./utils/assert_expected_indicator_source_index_privileges");
var _get_transform_compite_query = require("./utils/get_transform_compite_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CreateSLO {
  constructor(scopedClusterClient, repository, internalSOClient, transformManager, summaryTransformManager, logger, spaceId, basePath, username) {
    this.scopedClusterClient = scopedClusterClient;
    this.repository = repository;
    this.internalSOClient = internalSOClient;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.logger = logger;
    this.spaceId = spaceId;
    this.basePath = basePath;
    this.username = username;
  }
  async execute(params) {
    const slo = this.toSLO(params);
    (0, _services.validateSLO)(slo);
    await Promise.all([this.assertSLOInexistant(slo), (0, _assert_expected_indicator_source_index_privileges.assertExpectedIndicatorSourceIndexPrivileges)(slo, this.scopedClusterClient.asCurrentUser)]);
    const rollbackOperations = [];
    const createPromise = this.repository.create(slo);
    rollbackOperations.push(() => this.repository.deleteById(slo.id, {
      ignoreNotFound: true
    }));
    const rollupTransformId = (0, _constants2.getSLOTransformId)(slo.id, slo.revision);
    const summaryTransformId = (0, _constants2.getSLOSummaryTransformId)(slo.id, slo.revision);
    try {
      const sloPipelinePromise = this.createPipeline((0, _sli_pipeline_template.getSLIPipelineTemplate)(slo, this.spaceId));
      rollbackOperations.push(() => this.deletePipeline((0, _constants2.getSLOPipelineId)(slo.id, slo.revision)));
      const rollupTransformPromise = this.transformManager.install(slo);
      rollbackOperations.push(() => this.transformManager.uninstall(rollupTransformId));
      const summaryPipelinePromise = this.createPipeline((0, _summary_pipeline_template.getSummaryPipelineTemplate)(slo, this.spaceId, this.basePath));
      rollbackOperations.push(() => this.deletePipeline((0, _constants2.getSLOSummaryPipelineId)(slo.id, slo.revision)));
      const summaryTransformPromise = this.summaryTransformManager.install(slo);
      rollbackOperations.push(() => this.summaryTransformManager.uninstall(summaryTransformId));
      const tempDocPromise = this.createTempSummaryDocument(slo);
      rollbackOperations.push(() => this.deleteTempSummaryDocument(slo));
      await Promise.all([createPromise, sloPipelinePromise, rollupTransformPromise, summaryPipelinePromise, summaryTransformPromise, tempDocPromise]);

      // transforms can only be started after the related pipelines are created
      await Promise.all([this.transformManager.start(rollupTransformId), this.summaryTransformManager.start(summaryTransformId)]);
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      this.logger.debug(`Cannot create the SLO [id: ${slo.id}, revision: ${slo.revision}]. Rolling back. ${err}`);
      await (0, _std.asyncForEach)(rollbackOperations.reverse(), async operation => {
        try {
          await operation();
        } catch (rollbackErr) {
          this.logger.debug(`Rollback operation failed. ${rollbackErr}`);
        }
      });
      if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
    return this.toResponse(slo);
  }
  async assertSLOInexistant(slo) {
    const findResponse = await this.internalSOClient.find({
      type: _saved_objects.SO_SLO_TYPE,
      perPage: 0,
      filter: `slo.attributes.id:(${slo.id})`,
      namespaces: [_constants.ALL_SPACES_ID]
    });
    const exists = findResponse.total > 0;
    if (exists) {
      throw new _errors.SLOIdConflict(`SLO [${slo.id}] already exists`);
    }
  }
  async createTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.index({
      index: _constants2.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      document: (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId, this.basePath),
      refresh: true
    }), {
      logger: this.logger
    });
  }
  async deleteTempSummaryDocument(slo) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asCurrentUser.delete({
      index: _constants2.SUMMARY_TEMP_INDEX_NAME,
      id: `slo-${slo.id}`,
      refresh: true
    }), {
      logger: this.logger
    });
  }
  async createPipeline(params) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.putPipeline(params), {
      logger: this.logger
    });
  }
  async deletePipeline(id) {
    return (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.ingest.deletePipeline({
      id
    }, {
      ignore: [404]
    }), {
      logger: this.logger
    });
  }
  async inspect(params) {
    const slo = this.toSLO(params);
    (0, _services.validateSLO)(slo);
    const rollUpTransform = await this.transformManager.inspect(slo);
    const rollUpPipeline = (0, _sli_pipeline_template.getSLIPipelineTemplate)(slo, this.spaceId);
    const summaryPipeline = (0, _summary_pipeline_template.getSummaryPipelineTemplate)(slo, this.spaceId, this.basePath);
    const summaryTransform = await this.summaryTransformManager.inspect(slo);
    const temporaryDoc = (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId, this.basePath);
    return {
      slo,
      rollUpPipeline,
      summaryPipeline,
      temporaryDoc,
      summaryTransform,
      rollUpTransform,
      rollUpTransformCompositeQuery: (0, _get_transform_compite_query.getTransformQueryComposite)(rollUpTransform),
      summaryTransformCompositeQuery: (0, _get_transform_compite_query.getTransformQueryComposite)(summaryTransform)
    };
  }
  toSLO(params) {
    var _params$id, _params$revision, _params$tags;
    const now = new Date();
    return {
      ...params,
      id: (_params$id = params.id) !== null && _params$id !== void 0 ? _params$id : (0, _uuid.v4)(),
      settings: (0, _lodash.merge)({
        syncDelay: new _models.Duration(1, _models.DurationUnit.Minute),
        frequency: new _models.Duration(1, _models.DurationUnit.Minute),
        preventInitialBackfill: false
      }, params.settings),
      revision: (_params$revision = params.revision) !== null && _params$revision !== void 0 ? _params$revision : 1,
      enabled: true,
      tags: (_params$tags = params.tags) !== null && _params$tags !== void 0 ? _params$tags : [],
      createdAt: now,
      updatedAt: now,
      createdBy: this.username,
      updatedBy: this.username,
      groupBy: !!params.groupBy ? params.groupBy : _sloSchema.ALL_VALUE,
      version: _constants2.SLO_MODEL_VERSION
    };
  }
  toResponse(slo) {
    return {
      id: slo.id
    };
  }
}
exports.CreateSLO = CreateSLO;