"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetrySecurityListTaskConfig = createTelemetrySecurityListTaskConfig;
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _constants = require("../constants");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetrySecurityListTaskConfig(maxTelemetryBatch) {
  const taskName = 'Security Solution Lists Telemetry';
  const taskType = 'security:telemetry-lists';
  return {
    type: taskType,
    title: taskName,
    interval: '24h',
    timeout: '3m',
    version: '1.0.0',
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('security_lists'), mdc);
      const trace = taskMetricsService.start(taskType);
      log.debug('Running telemetry task');
      const usageCollector = sender.getTelemetryUsageCluster();
      const usageLabelPrefix = ['security_telemetry', 'lists'];
      try {
        let trustedApplicationsCount = 0;
        let endpointExceptionsCount = 0;
        let endpointEventFiltersCount = 0;
        const [clusterInfoPromise, licenseInfoPromise] = await Promise.allSettled([receiver.fetchClusterInfo(), receiver.fetchLicenseInfo()]);
        const clusterInfo = clusterInfoPromise.status === 'fulfilled' ? clusterInfoPromise.value : {};
        const licenseInfo = licenseInfoPromise.status === 'fulfilled' ? licenseInfoPromise.value : {};
        const FETCH_VALUE_LIST_META_DATA_INTERVAL_IN_HOURS = 24;

        // Lists Telemetry: Trusted Applications
        const trustedApps = await receiver.fetchTrustedApplications();
        if (trustedApps !== null && trustedApps !== void 0 && trustedApps.data) {
          const trustedAppsJson = (0, _helpers.templateExceptionList)(trustedApps.data, clusterInfo, licenseInfo, _constants.LIST_TRUSTED_APPLICATION);
          trustedApplicationsCount = trustedAppsJson.length;
          log.debug('Trusted Apps', {
            trusted_apps_count: trustedApplicationsCount
          });
          usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
            counterName: (0, _helpers.createUsageCounterLabel)(usageLabelPrefix),
            counterType: 'trusted_apps_count',
            incrementBy: trustedApplicationsCount
          });
          const batches = (0, _helpers.batchTelemetryRecords)(trustedAppsJson, maxTelemetryBatch);
          for (const batch of batches) {
            await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_LISTS, batch);
          }
        }

        // Lists Telemetry: Endpoint Exceptions

        const epExceptions = await receiver.fetchEndpointList(_securitysolutionListConstants.ENDPOINT_LIST_ID);
        if (epExceptions !== null && epExceptions !== void 0 && epExceptions.data) {
          const epExceptionsJson = (0, _helpers.templateExceptionList)(epExceptions.data, clusterInfo, licenseInfo, _constants.LIST_ENDPOINT_EXCEPTION);
          endpointExceptionsCount = epExceptionsJson.length;
          log.debug('EP Exceptions', {
            ep_exceptions_count: endpointExceptionsCount
          });
          usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
            counterName: (0, _helpers.createUsageCounterLabel)(usageLabelPrefix),
            counterType: 'endpoint_exceptions_count',
            incrementBy: endpointExceptionsCount
          });
          const batches = (0, _helpers.batchTelemetryRecords)(epExceptionsJson, maxTelemetryBatch);
          for (const batch of batches) {
            await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_LISTS, batch);
          }
        }

        // Lists Telemetry: Endpoint Event Filters

        const epFilters = await receiver.fetchEndpointList(_securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.eventFilters.id);
        if (epFilters !== null && epFilters !== void 0 && epFilters.data) {
          const epFiltersJson = (0, _helpers.templateExceptionList)(epFilters.data, clusterInfo, licenseInfo, _constants.LIST_ENDPOINT_EVENT_FILTER);
          endpointEventFiltersCount = epFiltersJson.length;
          log.debug('EP Event Filters', {
            ep_filters_count: endpointEventFiltersCount
          });
          usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
            counterName: (0, _helpers.createUsageCounterLabel)(usageLabelPrefix),
            counterType: 'endpoint_event_filters_count',
            incrementBy: endpointEventFiltersCount
          });
          const batches = (0, _helpers.batchTelemetryRecords)(epFiltersJson, maxTelemetryBatch);
          for (const batch of batches) {
            await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_LISTS, batch);
          }
        }

        // Value list meta data
        const valueListResponse = await receiver.fetchValueListMetaData(FETCH_VALUE_LIST_META_DATA_INTERVAL_IN_HOURS);
        const valueListMetaData = (0, _helpers.formatValueListMetaData)(valueListResponse, clusterInfo, licenseInfo);
        if (valueListMetaData !== null && valueListMetaData !== void 0 && valueListMetaData.total_list_count) {
          await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_LISTS, [valueListMetaData]);
        }
        await taskMetricsService.end(trace);
        return trustedApplicationsCount + endpointExceptionsCount + endpointEventFiltersCount;
      } catch (error) {
        log.warn('Error running security lists task', (0, _helpers.withErrorMessage)(error));
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}