"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTranslatePanelNode = void 0;
var _langgraph = require("@langchain/langgraph");
var _comments = require("../../../../../common/task/util/comments");
var _resources = require("../../../../../../../../common/siem_migrations/dashboards/resources");
var _translate_panel = require("../../sub_graphs/translate_panel");
var _create_markdown_panel = require("../../helpers/markdown_panel/create_markdown_panel");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Number of panels to be processed concurrently per dashboard */
const DEFAULT_PANELS_CONCURRENCY = 4;
// This is a special node, it's goal is to use map-reduce to translate the dashboard panels in parallel.
// This is the recommended technique at the time of writing this code. LangGraph docs: https://langchain-ai.github.io/langgraphjs/how-tos/map-reduce/.
const getTranslatePanelNode = params => {
  const translatePanelSubGraph = (0, _translate_panel.getTranslatePanelGraph)(params);
  return {
    // Fan-in: the results of the individual panel translations are aggregated back into the overall dashboard state via state reducer.
    node: async ({
      index,
      ...nodeParams
    }) => {
      let translatedPanel;
      try {
        if (!nodeParams.parsed_panel.query) {
          throw new Error('Panel query is missing');
        }

        // Invoke the subgraph to translate the panel
        const output = await translatePanelSubGraph.invoke(nodeParams, {
          maxConcurrency: DEFAULT_PANELS_CONCURRENCY
        });
        if (!output.elastic_panel) {
          throw new Error('No panel visualization generated');
        }
        translatedPanel = {
          index,
          title: nodeParams.parsed_panel.title,
          data: output.elastic_panel,
          translation_result: output.translation_result,
          comments: output.comments
        };
      } catch (err) {
        const message = `Error translating panel: ${err.toString()}`;
        params.logger.error(message);
        translatedPanel = {
          index,
          title: nodeParams.parsed_panel.title,
          data: (0, _create_markdown_panel.createMarkdownPanel)(message, nodeParams.parsed_panel),
          comments: [(0, _comments.generateAssistantComment)(message)],
          error: err
        };
      }
      return {
        translated_panels: [translatedPanel]
      };
    },
    // Fan-out: `conditionalEdge` that Send all individual "translatePanel" to be executed in parallel
    conditionalEdge: state => {
      var _state$parsed_origina;
      // Pre-condition: `state.parsed_original_dashboard.panels` must not be empty, otherwise the execution will stop here
      const panels = (_state$parsed_origina = state.parsed_original_dashboard.panels) !== null && _state$parsed_origina !== void 0 ? _state$parsed_origina : [];
      return panels.map((panel, i) => {
        const resources = filterIdentifiedResources(state.original_dashboard.vendor, state.resources, panel);
        const description = state.panel_descriptions[panel.id];
        const translatePanelParams = {
          parsed_panel: panel,
          description,
          dashboard_description: state.description,
          resources,
          index: i
        };
        return new _langgraph.Send('translatePanel', translatePanelParams);
      });
    },
    subgraph: translatePanelSubGraph // Only for the diagram generation
  };
};

/**
 * This function filters the stored resource data that have been received for the entire dashboard,
 * and returns only the resources that have been identified for each specific panel query.
 */
exports.getTranslatePanelNode = getTranslatePanelNode;
function filterIdentifiedResources(vendor, resources, panel) {
  var _resources$macro, _resources$lookup;
  const resourceIdentifier = new _resources.DashboardResourceIdentifier(vendor);
  const identifiedResources = resourceIdentifier.fromQuery(panel.query);
  const {
    macros,
    lookups
  } = identifiedResources.reduce((acc, {
    type,
    name
  }) => {
    if (type === 'macro') {
      acc.macros.push(name);
    } else if (type === 'lookup') {
      acc.lookups.push(name);
    }
    return acc;
  }, {
    macros: [],
    lookups: []
  });
  const filteredResources = {};
  const macro = (_resources$macro = resources.macro) === null || _resources$macro === void 0 ? void 0 : _resources$macro.filter(m => macros.includes(m.name));
  if (macro !== null && macro !== void 0 && macro.length) {
    filteredResources.macro = macro;
  }
  const lookup = (_resources$lookup = resources.lookup) === null || _resources$lookup === void 0 ? void 0 : _resources$lookup.filter(l => lookups.includes(l.name));
  if (lookup !== null && lookup !== void 0 && lookup.length) {
    filteredResources.lookup = lookup;
  }
  return filteredResources;
}