"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createInitialisationSourcesService = void 0;
var _lodash = require("lodash");
var _utils = require("../../../../../common/entity_analytics/privileged_user_monitoring/utils");
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _data_sources = require("../data_sources");
var _actions = require("../auditing/actions");
var _saved_objects = require("../saved_objects");
var _audit_logger = require("../audit_logger");
var _logger = require("../logger");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createInitialisationSourcesService = deps => {
  return async function upsertSources(namespace) {
    const logger = (0, _logger.createPrivMonLogger)(deps.logger, namespace);
    const auditLogger = (0, _audit_logger.createPrivMonAuditLogger)(deps.auditLogger);
    const index = (0, _utils.getPrivilegedMonitorUsersIndex)(namespace);
    try {
      // required sources to initialize privileged monitoring engine
      const requiredInitSources = buildRequiredSources(namespace, index);
      const existing = await deps.descriptorClient.findAll({});

      // create all sources, if none exist already
      if (existing.length === 0) {
        await deps.descriptorClient.bulkCreate(requiredInitSources);
        logger.log('debug', `Created all ${requiredInitSources.length} default sources`);
        return;
      }
      const requiredIntegrationNames = requiredInitSources.map(({
        name
      }) => name).sort();
      const installedIntegration = await deps.descriptorClient.findByQuery(buildFilterByIntegrationNames(requiredIntegrationNames));
      const installedIntegrationsNames = installedIntegration.map(({
        name
      }) => name).sort();
      if (!(0, _lodash.isEqual)(requiredIntegrationNames, installedIntegrationsNames)) {
        const {
          created,
          updated,
          results
        } = await deps.descriptorClient.bulkUpsert(requiredInitSources);
        logger.log('debug', `Privilege Monitoring sources upsert - created: ${created}, updated: ${updated}, processed: ${results.length}.`);
      }
    } catch (error) {
      logger.log('error', `Failed to create default index source for privilege monitoring: ${error.message}`);
      auditLogger.log(_actions.PrivilegeMonitoringEngineActions.INIT, _entity_analytics.MonitoringEngineComponentResourceEnum.privmon_engine, 'Failed to create default index source for privilege monitoring', error);
    }
  };
};
exports.createInitialisationSourcesService = createInitialisationSourcesService;
const getLastFullSyncMarkersIndex = (namespace, integration) => {
  if (integration === 'entityanalytics_ad') {
    return (0, _data_sources.getStreamPatternFor)(integration, namespace);
  }
  // okta has a dedicated index for last full sync markers
  return (0, _data_sources.oktaLastFullSyncMarkersIndex)(namespace);
};
const makeIntegrationSource = (namespace, integration) => ({
  type: 'entity_analytics_integration',
  managed: true,
  indexPattern: (0, _data_sources.getStreamPatternFor)(integration, namespace),
  name: (0, _data_sources.integrationsSourceIndex)(namespace, integration),
  matchers: (0, _data_sources.getMatchersFor)(integration),
  integrationName: integration,
  integrations: {
    syncMarkerIndex: getLastFullSyncMarkersIndex(namespace, integration)
  }
});
function buildRequiredSources(namespace, indexPattern) {
  const integrations = _data_sources.INTEGRATION_TYPES.map(integration => makeIntegrationSource(namespace, integration));
  return [makeDefaultIndexSource(namespace, indexPattern), ...integrations];
}
const makeDefaultIndexSource = (namespace, name) => ({
  type: 'index',
  managed: true,
  indexPattern: (0, _utils.defaultMonitoringUsersIndex)(namespace),
  name
});
const buildFilterByIntegrationNames = requiredIntegrationNames => {
  return requiredIntegrationNames.map(name => `${_saved_objects.monitoringEntitySourceTypeName}.attributes.name: ${name}`).join(' OR ');
};