"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildPrivilegedSearchBody = exports.buildMatcherScript = exports.applyPrivilegedUpdates = void 0;
var _upsert = require("../../../bulk/upsert");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Build painless script for matcher
 * @param matcher - matcher object containing fields and values to match against
 * @returns
 */
const buildMatcherScript = matcher => {
  if (!matcher || matcher.fields.length === 0 || matcher.values.length === 0) {
    throw new Error('Invalid matcher: fields and values must be defined and non-empty');
  }
  return {
    lang: 'painless',
    params: {
      matcher_fields: matcher.fields,
      matcher_values: matcher.values
    },
    source: `
      for (def f : params.matcher_fields) {
        if (doc.containsKey(f) && !doc[f].empty) {
          for (def v : doc[f]) {
            if (params.matcher_values.contains(v)) return true;
          }
        }
      }
      return false;
    `
  };
};

/**
 * Building privileged search body for matchers
 */
exports.buildMatcherScript = buildMatcherScript;
const buildPrivilegedSearchBody = (script, timeGte, matchersField, afterKey, pageSize = 100) => ({
  size: 0,
  query: {
    range: {
      '@timestamp': {
        gte: timeGte,
        lte: 'now'
      }
    }
  },
  aggs: {
    privileged_user_status_since_last_run: {
      composite: {
        size: pageSize,
        sources: [{
          username: {
            terms: {
              field: 'user.name'
            }
          }
        }],
        ...(afterKey ? {
          after: afterKey
        } : {})
      },
      aggs: {
        latest_doc_for_user: {
          top_hits: {
            size: 1,
            sort: [{
              '@timestamp': {
                order: 'desc'
              }
            }],
            script_fields: {
              'user.is_privileged': {
                script
              }
            },
            _source: {
              includes: ['user', matchersField, '@timestamp']
            }
          }
        }
      }
    }
  }
});
exports.buildPrivilegedSearchBody = buildPrivilegedSearchBody;
const applyPrivilegedUpdates = async ({
  dataClient,
  users,
  source
}) => {
  if (users.length === 0) return;
  const chunkSize = 500;
  const esClient = dataClient.deps.clusterClient.asCurrentUser;
  const opsForIntegration = (0, _upsert.makeIntegrationOpsBuilder)(dataClient);
  try {
    for (let start = 0; start < users.length; start += chunkSize) {
      const chunk = users.slice(start, start + chunkSize);
      const operations = opsForIntegration(chunk, source);
      if (operations.length > 0) {
        const resp = await esClient.bulk({
          refresh: 'wait_for',
          body: operations
        });
        const errors = (0, _utils.getErrorFromBulkResponse)(resp);
        dataClient.log('error', (0, _utils.errorsMsg)(errors));
      }
    }
  } catch (error) {
    dataClient.log('error', `Error applying privileged updates: ${error.message}`);
  }
};
exports.applyPrivilegedUpdates = applyPrivilegedUpdates;