"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPlatformPipelineStatus = exports.deletePlatformPipeline = exports.createPlatformPipeline = exports.ENRICH_FIELD = void 0;
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _ingest_processor_steps = require("./ingest_processor_steps");
var _enrich_policy = require("./enrich_policy");
var _field_retention = require("../field_retention");
var _dynamic_retention = require("../field_retention/dynamic_retention");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPlatformPipelineId = descriptionId => {
  return `${descriptionId}-latest@platform`;
};

// the field that the enrich processor writes to
const ENRICH_FIELD = exports.ENRICH_FIELD = 'historical';

/**
 * Builds the ingest pipeline for the field retention policy.
 * Broadly the pipeline enriches the entity with the field retention enrich policy,
 * then applies the field retention policy to the entity fields, and finally removes
 * the enrich field and any empty fields.
 *
 * While developing, be sure to set debugMode to true this will keep the enrich field
 * and the context field in the document to help with debugging.
 */
const buildIngestPipeline = ({
  allEntityFields,
  debugMode,
  namespace,
  description
}) => {
  const enrichPolicyName = (0, _enrich_policy.getFieldRetentionEnrichPolicyName)({
    namespace,
    entityType: description.entityType,
    version: description.version
  });
  const processors = [{
    set: {
      field: '@timestamp',
      value: '{{entity.last_seen_timestamp}}'
    }
  }, {
    set: {
      field: 'entity.name',
      override: false,
      value: `{{${description.identityField}}}`
    }
  }, ...(debugMode ? [{
    set: {
      field: 'debug.collected',
      value: '{{collected.metadata}}'
    }
  }, {
    set: {
      field: 'debug._source',
      value: '{{_source}}'
    }
  }] : []), {
    enrich: {
      policy_name: enrichPolicyName,
      field: description.identityField,
      target_field: ENRICH_FIELD
    }
  }, ...(0, _ingest_processor_steps.getDotExpanderSteps)(allEntityFields), ...description.fields.map(field => (0, _field_retention.fieldOperatorToIngestProcessor)(field, {
    enrichField: ENRICH_FIELD
  })), ...(0, _ingest_processor_steps.getRemoveEmptyFieldSteps)([...allEntityFields, 'asset', `${description.entityType}.risk`]), (0, _ingest_processor_steps.removeEntityDefinitionFieldsStep)(), ...(description.dynamic ? [(0, _dynamic_retention.dynamicNewestRetentionSteps)(description.fields.map(field => field.destination))] : []), ...(Array.isArray(description.pipeline) ? description.pipeline : []), ...(debugMode ? [(0, _ingest_processor_steps.debugDeepCopyContextStep)()] : [{
    remove: {
      ignore_failure: true,
      field: ENRICH_FIELD
    }
  }])];

  // if the pipeline is a function, we call it with the default processors
  // this allows the entity definition to add custom processors to the pipeline or modify the default ones
  return typeof description.pipeline === 'function' ? description.pipeline(processors) : processors;
};

// developing the pipeline is a bit tricky, so we have a debug mode
// set  xpack.securitySolution.entityAnalytics.entityStore.developer.pipelineDebugMode
// to true to keep the enrich field and the context field in the document to help with debugging.
const createPlatformPipeline = async ({
  logger,
  esClient,
  debugMode,
  description,
  options
}) => {
  const allEntityFields = description.fields.map(({
    destination
  }) => destination);
  const pipeline = {
    id: getPlatformPipelineId(description.id),
    _meta: {
      managed_by: 'entity_store',
      managed: true
    },
    description: `Ingest pipeline for entity definition ${description.id}`,
    processors: buildIngestPipeline({
      namespace: options.namespace,
      description,
      version: description.version,
      allEntityFields,
      debugMode
    })
  };
  logger.debug(`Attempting to create pipeline: ${JSON.stringify(pipeline)}`);
  await esClient.ingest.putPipeline(pipeline);
};
exports.createPlatformPipeline = createPlatformPipeline;
const deletePlatformPipeline = ({
  description,
  logger,
  esClient
}) => {
  const pipelineId = getPlatformPipelineId(description.id);
  logger.debug(`Attempting to delete pipeline: ${pipelineId}`);
  return esClient.ingest.deletePipeline({
    id: pipelineId
  }, {
    ignore: [404]
  });
};
exports.deletePlatformPipeline = deletePlatformPipeline;
const getPlatformPipelineStatus = async ({
  engineId,
  esClient
}) => {
  const pipelineId = getPlatformPipelineId(engineId);
  const pipeline = await esClient.ingest.getPipeline({
    id: pipelineId
  }, {
    ignore: [404]
  });
  return {
    id: pipelineId,
    installed: !!pipeline[pipelineId],
    resource: _entity_analytics.EngineComponentResourceEnum.ingest_pipeline
  };
};
exports.getPlatformPipelineStatus = getPlatformPipelineStatus;