"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetCriticalityPublicCSVUploadRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _configSchema = require("@kbn/config-schema");
var _papaparse = _interopRequireDefault(require("papaparse"));
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _asset_criticality = require("../../../../../common/entity_analytics/asset_criticality");
var _constants = require("../../../../../common/constants");
var _check_and_init_asset_criticality_resources = require("../check_and_init_asset_criticality_resources");
var _transform_csv_to_upsert_records = require("../transform_csv_to_upsert_records");
var _events = require("../../../telemetry/event_based/events");
var _audit = require("../audit");
var _audit2 = require("../../audit");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const assetCriticalityPublicCSVUploadRoute = (router, logger, config, getStartServices) => {
  router.versioned.post({
    access: 'public',
    path: _constants.ASSET_CRITICALITY_PUBLIC_CSV_UPLOAD_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    },
    options: {
      body: {
        output: 'stream',
        accepts: 'multipart/form-data',
        maxBytes: _asset_criticality.CRITICALITY_CSV_MAX_SIZE_BYTES_WITH_TOLERANCE
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: _configSchema.schema.object({
          file: _configSchema.schema.stream()
        })
      }
    }
  }, async (context, request, response) => {
    var _securitySolution$get;
    const {
      errorRetries,
      maxBulkRequestBodySizeBytes
    } = config.entityAnalytics.assetCriticality.csvUpload;
    const securitySolution = await context.securitySolution;
    (_securitySolution$get = securitySolution.getAuditLogger()) === null || _securitySolution$get === void 0 ? void 0 : _securitySolution$get.log({
      message: 'User attempted to assign many asset criticalities via file upload',
      event: {
        action: _audit.AssetCriticalityAuditActions.ASSET_CRITICALITY_BULK_UPDATE,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.CREATION,
        outcome: _audit2.AUDIT_OUTCOME.UNKNOWN
      }
    });
    const start = new Date();
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const [coreStart] = await getStartServices();
    const telemetry = coreStart.analytics;
    try {
      await (0, _check_and_init_asset_criticality_resources.checkAndInitAssetCriticalityResources)(context, logger);
      const assetCriticalityClient = securitySolution.getAssetCriticalityDataClient();
      const fileStream = request.body.file;
      logger.debug(`Parsing asset criticality CSV file ${fileStream.hapi.filename}`);
      const csvStream = _papaparse.default.parse(_papaparse.default.NODE_STREAM_INPUT, {
        header: false,
        dynamicTyping: true,
        skipEmptyLines: true
      });
      const recordsStream = fileStream.pipe(csvStream).pipe((0, _transform_csv_to_upsert_records.transformCSVToUpsertRecords)(config.experimentalFeatures));
      const {
        errors,
        stats
      } = await assetCriticalityClient.bulkUpsertFromStream({
        recordsStream,
        retries: errorRetries,
        flushBytes: maxBulkRequestBodySizeBytes,
        streamIndexStart: 1 // It is the first line number
      });
      const end = new Date();
      const tookMs = end.getTime() - start.getTime();
      logger.debug(() => `Asset criticality CSV upload completed in ${tookMs}ms ${JSON.stringify(stats)}`);

      // type assignment here to ensure that the response body stays in sync with the API schema
      const resBody = {
        errors,
        stats
      };
      const [eventType, event] = (0, _events.createAssetCriticalityProcessedFileEvent)({
        startTime: start,
        endTime: end,
        result: stats
      });
      telemetry.reportEvent(eventType, event);
      return response.ok({
        body: resBody
      });
    } catch (e) {
      logger.error(`Error during asset criticality csv upload: ${e}`);
      try {
        const end = new Date();
        const [eventType, event] = (0, _events.createAssetCriticalityProcessedFileEvent)({
          startTime: start,
          endTime: end
        });
        telemetry.reportEvent(eventType, event);
      } catch (error) {
        logger.error(`Error reporting telemetry event: ${error}`);
      }
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.assetCriticalityPublicCSVUploadRoute = assetCriticalityPublicCSVUploadRoute;