"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateExcludedDocuments = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updates the list of excluded documents
 * The last document ID in the multiple results is not excluded to avoid missing alerts in case of mv_expand usage or reaching max_signals limit
 * It is to ensure that in the next page we will be able to catch the rest of expanded values or alerts not created when reached max_signals limit
 */
const updateExcludedDocuments = ({
  excludedDocuments,
  sourceDocuments,
  results,
  isRuleAggregating,
  aggregatableTimestampField,
  searchExhausted
}) => {
  // aggregating queries do not have event _id, so we will not exclude any documents
  if (isRuleAggregating) {
    return;
  }
  const totalSourceDocuments = Object.keys(sourceDocuments).reduce((acc, index) => acc + sourceDocuments[index].length, 0);
  const documentIds = Object.keys(sourceDocuments);
  if (totalSourceDocuments !== 1 && !searchExhausted) {
    var _results$at, _results$at2;
    const lastId = (_results$at = results.at(-1)) === null || _results$at === void 0 ? void 0 : _results$at._id;
    const lastIndex = (_results$at2 = results.at(-1)) === null || _results$at2 === void 0 ? void 0 : _results$at2._index;

    // if single document is returned(same id, same index across all results), we will exclude it
    const excludeSingleDocument = documentIds.length === 1 && results.reduce((acc, doc) => {
      acc.add(doc._index);
      return acc;
    }, new Set()).size === 1;
    if (lastId) {
      if (lastIndex === undefined) {
        var _sourceDocuments$last;
        (_sourceDocuments$last = sourceDocuments[lastId]) === null || _sourceDocuments$last === void 0 ? void 0 : _sourceDocuments$last.pop();
      } else {
        sourceDocuments[lastId] = sourceDocuments[lastId].filter(doc => {
          return excludeSingleDocument ? doc._index === lastIndex : doc._index !== lastIndex;
        });
      }
    }
  }
  addToExcludedDocuments(excludedDocuments, sourceDocuments, documentIds, aggregatableTimestampField);
};
exports.updateExcludedDocuments = updateExcludedDocuments;
const addToExcludedDocuments = (excludedDocuments, sourceDocuments, documentIds, aggregatableTimestampField) => {
  for (const documentId of documentIds) {
    const documents = sourceDocuments[documentId];
    documents.forEach(document => {
      var _document$fields, _document$fields$aggr;
      if (!excludedDocuments[document._index]) {
        excludedDocuments[document._index] = [];
      }
      excludedDocuments[document._index].push({
        id: documentId,
        timestamp: (_document$fields = document.fields) === null || _document$fields === void 0 ? void 0 : (_document$fields$aggr = _document$fields[aggregatableTimestampField]) === null || _document$fields$aggr === void 0 ? void 0 : _document$fields$aggr[0]
      });
    });
  }
};