"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esqlExecutor = void 0;
var _perf_hooks = require("perf_hooks");
var _lodash = require("lodash");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _build_esql_search_request = require("./build_esql_search_request");
var _esql_request = require("./esql_request");
var _wrap_esql_alerts = require("./wrap_esql_alerts");
var _wrap_suppressed_esql_alerts = require("./wrap_suppressed_esql_alerts");
var _bulk_create_suppressed_alerts_in_memory = require("../utils/bulk_create_suppressed_alerts_in_memory");
var _utils = require("./utils");
var _fetch_source_documents = require("./fetch_source_documents");
var _reason_formatters = require("../utils/reason_formatters");
var _get_data_tier_filter = require("../utils/get_data_tier_filter");
var _check_error_details = require("../utils/check_error_details");
var _log_cluster_shard_failures_esql = require("../utils/log_cluster_shard_failures_esql");
var _utils2 = require("../utils/utils");
var _with_security_span = require("../../../../utils/with_security_span");
var _get_is_alert_suppression_active = require("../utils/get_is_alert_suppression_active");
var _factories = require("../factories");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_EXCLUDED_DOCUMENTS = 100 * 1000;
const esqlExecutor = async ({
  sharedParams,
  services,
  state,
  licensing,
  scheduleNotificationResponseActionsService,
  ruleExecutionTimeout
}) => {
  var _state$isLoggedReques;
  const {
    completeRule,
    tuple,
    primaryTimestamp,
    secondaryTimestamp,
    exceptionFilter,
    unprocessedExceptions,
    ruleExecutionLogger
  } = sharedParams;
  const loggedRequests = [];
  const ruleParams = completeRule.ruleParams;
  const isLoggedRequestsEnabled = (_state$isLoggedReques = state === null || state === void 0 ? void 0 : state.isLoggedRequestsEnabled) !== null && _state$isLoggedReques !== void 0 ? _state$isLoggedReques : false;
  return (0, _with_security_span.withSecuritySpan)('esqlExecutor', async () => {
    const result = (0, _utils2.createSearchAfterReturnType)();
    const dataTiersFilters = await (0, _get_data_tier_filter.getDataTierFilter)({
      uiSettingsClient: services.uiSettingsClient
    });
    const isRuleAggregating = (0, _securitysolutionUtils.computeIsESQLQueryAggregating)(ruleParams.query);
    const hasMvExpand = (0, _securitysolutionUtils.getMvExpandFields)(ruleParams.query).length > 0;
    // since pagination is not supported in ES|QL, we will use tuple.maxSignals + 1 to determine if search results are exhausted
    const size = tuple.maxSignals + 1;
    const excludedDocuments = (0, _utils.initiateExcludedDocuments)({
      state,
      isRuleAggregating,
      tuple,
      hasMvExpand,
      query: ruleParams.query
    });

    /**
     * ES|QL returns results as a single page, max size of 10,000
     * To mitigate this, we will use the maxSignals as a page size
     * Wll keep track of the earlier found document ids and will exclude them in subsequent requests
     * to avoid duplicates.
     * This is a workaround until pagination is supported in ES|QL
     * Since aggregating queries do not produce event ids, we will not exclude them.
     * All alerts for aggregating queries are unique anyway
     */
    let iteration = 0;
    try {
      while (result.createdSignalsCount <= tuple.maxSignals) {
        const totalExcludedDocumentsLength = Object.values(excludedDocuments).reduce((acc, docs) => acc + docs.length, 0);
        if (totalExcludedDocumentsLength > MAX_EXCLUDED_DOCUMENTS) {
          result.warningMessages.push(`Excluded documents exceeded the limit of ${MAX_EXCLUDED_DOCUMENTS}, some alerts might not have been created. Consider reducing the lookback time for the rule.`);
          break;
        }
        const esqlRequest = (0, _build_esql_search_request.buildEsqlSearchRequest)({
          query: ruleParams.query,
          from: tuple.from.toISOString(),
          to: tuple.to.toISOString(),
          size,
          filters: dataTiersFilters,
          primaryTimestamp,
          secondaryTimestamp,
          exceptionFilter,
          excludedDocuments,
          ruleExecutionTimeout
        });
        const esqlQueryString = {
          drop_null_columns: true,
          // allow_partial_results is true by default, but we need to set it to false for aggregating queries
          allow_partial_results: !isRuleAggregating
        };
        const hasLoggedRequestsReachedLimit = iteration >= 2;
        ruleExecutionLogger.debug(`ES|QL query request: ${JSON.stringify(esqlRequest)}`);
        const exceptionsWarning = (0, _utils2.getUnprocessedExceptionsWarnings)(unprocessedExceptions);
        if (exceptionsWarning) {
          result.warningMessages.push(exceptionsWarning);
        }
        const esqlSignalSearchStart = _perf_hooks.performance.now();
        const response = await (0, _esql_request.performEsqlRequest)({
          esClient: services.scopedClusterClient.asCurrentUser,
          requestBody: esqlRequest,
          requestQueryParams: esqlQueryString,
          shouldStopExecution: services.shouldStopExecution,
          ruleExecutionLogger,
          loggedRequests: isLoggedRequestsEnabled ? loggedRequests : undefined
        });
        (0, _log_cluster_shard_failures_esql.logClusterShardFailuresEsql)({
          response,
          result
        });
        const esqlSearchDuration = _perf_hooks.performance.now() - esqlSignalSearchStart;
        result.searchAfterTimes.push((0, _utils2.makeFloatString)(esqlSearchDuration));
        ruleExecutionLogger.debug(`ES|QL query request for ${iteration} iteration took: ${esqlSearchDuration}ms`);
        const results = response.values.map(row => (0, _utils.rowToDocument)(response.columns, row));
        const index = (0, _securitysolutionUtils.getIndexListFromEsqlQuery)(completeRule.ruleParams.query);
        const sourceDocuments = await (0, _fetch_source_documents.fetchSourceDocuments)({
          esClient: services.scopedClusterClient.asCurrentUser,
          results,
          index,
          isRuleAggregating,
          loggedRequests: isLoggedRequestsEnabled ? loggedRequests : undefined,
          hasLoggedRequestsReachedLimit,
          runtimeMappings: sharedParams.runtimeMappings,
          excludedDocuments
        });
        const isAlertSuppressionActive = await (0, _get_is_alert_suppression_active.getIsAlertSuppressionActive)({
          alertSuppression: completeRule.ruleParams.alertSuppression,
          licensing
        });
        const {
          expandedFieldsInResponse: expandedFields
        } = (0, _utils.getMvExpandUsage)(response.columns, completeRule.ruleParams.query);
        const syntheticHits = results.map(document => {
          const {
            _id,
            _version,
            _index,
            ...esqlResult
          } = document;
          const sourceDocument = (0, _utils.getSourceDocument)(sourceDocuments, _id, _index);
          // when mv_expand command present we must clone source, since the reference will be used multiple times
          const source = hasMvExpand ? (0, _lodash.cloneDeep)(sourceDocument === null || sourceDocument === void 0 ? void 0 : sourceDocument._source) : sourceDocument === null || sourceDocument === void 0 ? void 0 : sourceDocument._source;
          return {
            _source: (0, _utils.mergeEsqlResultInSource)(source, esqlResult),
            fields: sourceDocument === null || sourceDocument === void 0 ? void 0 : sourceDocument.fields,
            _id: _id !== null && _id !== void 0 ? _id : '',
            _index: _index || (sourceDocument === null || sourceDocument === void 0 ? void 0 : sourceDocument._index) || '',
            _version: sourceDocument === null || sourceDocument === void 0 ? void 0 : sourceDocument._version
          };
        });
        if (isAlertSuppressionActive && (0, _get_is_alert_suppression_active.alertSuppressionTypeGuard)(completeRule.ruleParams.alertSuppression)) {
          const wrapSuppressedHits = events => (0, _wrap_suppressed_esql_alerts.wrapSuppressedEsqlAlerts)({
            sharedParams,
            events,
            isRuleAggregating,
            expandedFields
          });
          const bulkCreateResult = await (0, _bulk_create_suppressed_alerts_in_memory.bulkCreateSuppressedAlertsInMemory)({
            sharedParams,
            enrichedEvents: syntheticHits,
            toReturn: result,
            services,
            alertSuppression: completeRule.ruleParams.alertSuppression,
            wrapSuppressedHits,
            buildReasonMessage: _reason_formatters.buildReasonMessageForEsqlAlert,
            mergeSourceAndFields: true,
            // passing 1 here since ES|QL does not support pagination
            maxNumberOfAlertsMultiplier: 1
          });
          ruleExecutionLogger.debug(`Created ${bulkCreateResult.createdItemsCount} alerts. Suppressed ${bulkCreateResult.suppressedItemsCount} alerts`);
          (0, _utils.updateExcludedDocuments)({
            excludedDocuments,
            sourceDocuments,
            results,
            isRuleAggregating,
            aggregatableTimestampField: sharedParams.aggregatableTimestampField,
            searchExhausted: results.length < size
          });
          if (bulkCreateResult.alertsWereTruncated) {
            result.warningMessages.push((0, _utils2.getSuppressionMaxSignalsWarning)());
            break;
          }
        } else {
          const wrappedAlerts = (0, _wrap_esql_alerts.wrapEsqlAlerts)({
            sharedParams,
            events: syntheticHits,
            isRuleAggregating,
            expandedFields
          });
          const bulkCreateResult = await (0, _factories.bulkCreate)({
            wrappedAlerts,
            services,
            sharedParams,
            maxAlerts: tuple.maxSignals - result.createdSignalsCount
          });
          (0, _utils2.addToSearchAfterReturn)({
            current: result,
            next: bulkCreateResult
          });
          ruleExecutionLogger.debug(`Created ${bulkCreateResult.createdItemsCount} alerts`);
          (0, _utils.updateExcludedDocuments)({
            excludedDocuments,
            sourceDocuments,
            results,
            isRuleAggregating,
            aggregatableTimestampField: sharedParams.aggregatableTimestampField,
            searchExhausted: results.length < size
          });
          if (bulkCreateResult.alertsWereTruncated) {
            result.warningMessages.push((0, _utils2.getMaxSignalsWarning)());
            break;
          }
        }
        scheduleNotificationResponseActionsService({
          signals: result.createdSignals,
          signalsCount: result.createdSignalsCount,
          responseActions: completeRule.ruleParams.responseActions
        });

        // no more results will be found
        if (response.values.length < size) {
          ruleExecutionLogger.debug(`End of search: Found ${response.values.length} results with page size ${size}`);
          break;
        }
        iteration++;
      }
    } catch (error) {
      if ((0, _check_error_details.checkErrorDetails)(error).isUserError) {
        result.userError = true;
      }
      result.errors.push(error.message);
      result.success = false;
    }
    return {
      ...result,
      state: {
        ...state,
        excludedDocuments,
        lastQuery: hasMvExpand ? ruleParams.query : undefined // lastQuery is only relevant for mv_expand queries
      },
      ...(isLoggedRequestsEnabled ? {
        loggedRequests
      } : {})
    };
  });
};
exports.esqlExecutor = esqlExecutor;