"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.previewRulesRoute = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _uuid = require("uuid");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _common = require("@kbn/alerting-plugin/common");
var _zodHelpers = require("@kbn/zod-helpers");
var _lib = require("@kbn/alerting-plugin/server/lib");
var _constants = require("../../../../../../common/constants");
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _rule_monitoring = require("../../../../../../common/api/detection_engine/rule_monitoring");
var _detection_engine = require("../../../../../../common/api/detection_engine");
var _utils = require("../../../routes/utils");
var _preview_rule_execution_logger = require("./preview_rule_execution_logger");
var _utils2 = require("../../../rule_types/utils/utils");
var _authz = require("../../../../machine_learning/authz");
var _validation = require("../../../../machine_learning/validation");
var _route_limited_concurrency_tag = require("../../../../../utils/route_limited_concurrency_tag");
var _alert_instance_factory_stub = require("./alert_instance_factory_stub");
var _rule_types = require("../../../rule_types");
var _create_security_rule_type_wrapper = require("../../../rule_types/create_security_rule_type_wrapper");
var _utility_types = require("../../../../../../common/utility_types");
var _wrap_scoped_cluster_client = require("./wrap_scoped_cluster_client");
var _wrap_search_source_client = require("./wrap_search_source_client");
var _apply_rule_defaults = require("../../../rule_management/logic/detection_rules_client/mergers/apply_rule_defaults");
var _convert_rule_response_to_alerting_rule = require("../../../rule_management/logic/detection_rules_client/converters/convert_rule_response_to_alerting_rule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PREVIEW_TIMEOUT_SECONDS = 60;
const MAX_ROUTE_CONCURRENCY = 10;
const previewRulesRoute = (router, config, ml, security, securityRuleTypeOptions, previewRuleDataClient, getStartServices, logger, isServerless) => {
  router.versioned.post({
    path: _constants.DETECTION_ENGINE_RULES_PREVIEW,
    access: 'public',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      tags: [(0, _route_limited_concurrency_tag.routeLimitedConcurrencyTag)(MAX_ROUTE_CONCURRENCY)]
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_detection_engine.RulePreviewRequestBody),
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_detection_engine.RulePreviewRequestQuery)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const validationErrors = (0, _rule_management.validateCreateRuleProps)(request.body);
    const coreContext = await context.core;
    if (validationErrors.length) {
      return siemResponse.error({
        statusCode: 400,
        body: validationErrors
      });
    }
    try {
      var _security$authc$getCu;
      const [, {
        data,
        security: securityService,
        share,
        dataViews
      }] = await getStartServices();
      const searchSourceClient = await data.search.searchSource.asScoped(request);
      const savedObjectsClient = coreContext.savedObjects.client;
      const siemClient = (await context.securitySolution).getAppClient();
      const actionsClient = (await context.actions).getActionsClient();
      const timeframeEnd = request.body.timeframeEnd;
      let invocationCount = request.body.invocationCount;
      if (invocationCount < 1) {
        return response.ok({
          body: {
            logs: [{
              errors: ['Invalid invocation count'],
              warnings: [],
              duration: 0
            }],
            previewId: undefined,
            isAborted: undefined
          }
        });
      }
      const internalRule = (0, _convert_rule_response_to_alerting_rule.convertRuleResponseToAlertingRule)((0, _apply_rule_defaults.applyRuleDefaults)(request.body), actionsClient);
      const previewRuleParams = internalRule.params;
      const mlAuthz = (0, _authz.buildMlAuthz)({
        license: (await context.licensing).license,
        ml,
        request,
        savedObjectsClient
      });
      (0, _validation.throwAuthzError)(await mlAuthz.validateRuleType(internalRule.params.type));
      const listsContext = await context.lists;
      await (listsContext === null || listsContext === void 0 ? void 0 : listsContext.getExceptionListClient().createEndpointList());
      const spaceId = siemClient.getSpaceId();
      const previewId = (0, _uuid.v4)();
      const username = security === null || security === void 0 ? void 0 : (_security$authc$getCu = security.authc.getCurrentUser(request)) === null || _security$authc$getCu === void 0 ? void 0 : _security$authc$getCu.username;
      const loggedStatusChanges = [];
      const previewRuleExecutionLogger = (0, _preview_rule_execution_logger.createPreviewRuleExecutionLogger)(loggedStatusChanges);
      const runState = {
        isLoggedRequestsEnabled: request.query.enable_logged_requests
      };
      const logs = [];
      let isAborted = false;
      const {
        hasAllRequested
      } = await securityService.authz.checkPrivilegesWithRequest(request).atSpace(spaceId, {
        elasticsearch: {
          index: {
            [`${_constants.DEFAULT_PREVIEW_INDEX}-${spaceId}`]: ['read'],
            [`.internal${_constants.DEFAULT_PREVIEW_INDEX}-${spaceId}-*`]: ['read']
          },
          cluster: []
        }
      });
      if (!hasAllRequested) {
        return response.ok({
          body: {
            logs: [{
              errors: ['Missing "read" privileges for the ".preview.alerts-security.alerts" or ".internal.preview.alerts-security.alerts" indices. Without these privileges you cannot use the Rule Preview feature.'],
              warnings: [],
              duration: 0
            }],
            previewId: undefined,
            isAborted: undefined
          }
        });
      }
      const previewRuleTypeWrapper = (0, _create_security_rule_type_wrapper.createSecurityRuleTypeWrapper)({
        ...securityRuleTypeOptions,
        ruleDataClient: previewRuleDataClient,
        ruleExecutionLoggerFactory: previewRuleExecutionLogger.factory,
        isPreview: true
      });
      const runExecutors = async (securityRuleType, params) => {
        var _parseDuration;
        const ruleType = previewRuleTypeWrapper(securityRuleType);
        let statePreview = runState;
        let loggedRequests = [];
        const abortController = new AbortController();
        setTimeout(() => {
          abortController.abort();
          isAborted = true;
        }, PREVIEW_TIMEOUT_SECONDS * 1000);
        const startedAt = (0, _moment.default)(timeframeEnd);
        const parsedDuration = (_parseDuration = (0, _common.parseDuration)(internalRule.schedule.interval)) !== null && _parseDuration !== void 0 ? _parseDuration : 0;
        startedAt.subtract(_moment.default.duration(parsedDuration * (invocationCount - 1)));
        let previousStartedAt = null;
        const rule = {
          ...internalRule,
          id: previewId,
          createdAt: new Date(),
          createdBy: username !== null && username !== void 0 ? username : 'preview-created-by',
          producer: 'preview-producer',
          consumer: _constants.SERVER_APP_ID,
          enabled: true,
          revision: 0,
          ruleTypeId: ruleType.id,
          ruleTypeName: ruleType.name,
          updatedAt: new Date(),
          updatedBy: username !== null && username !== void 0 ? username : 'preview-updated-by',
          muteAll: false,
          snoozeSchedule: [],
          // In Security Solution, action params are typed as Record<string,
          // unknown>, which is a correct type for action params, but we
          // need to cast here to comply with the alerting types
          actions: internalRule.actions
        };
        let invocationStartTime;
        const dataViewsService = await dataViews.dataViewsServiceFactory(savedObjectsClient, coreContext.elasticsearch.client.asInternalUser);
        while (invocationCount > 0 && !isAborted) {
          invocationStartTime = (0, _moment.default)();
          ({
            state: statePreview,
            loggedRequests
          } = await ruleType.executor({
            executionId: (0, _uuid.v4)(),
            params,
            previousStartedAt,
            rule,
            services: {
              shouldWriteAlerts: () => true,
              shouldStopExecution: () => isAborted,
              alertsClient: null,
              alertFactory: {
                create: _alert_instance_factory_stub.alertInstanceFactoryStub,
                alertLimit: {
                  getValue: () => 1000,
                  setLimitReached: () => {}
                },
                done: () => ({
                  getRecoveredAlerts: () => []
                })
              },
              savedObjectsClient: coreContext.savedObjects.client,
              scopedClusterClient: (0, _wrap_scoped_cluster_client.wrapScopedClusterClient)({
                abortController,
                scopedClusterClient: coreContext.elasticsearch.client
              }),
              getSearchSourceClient: async () => (0, _wrap_search_source_client.wrapSearchSourceClient)({
                abortController,
                searchSourceClient
              }),
              getMaintenanceWindowIds: async () => [],
              uiSettingsClient: coreContext.uiSettings.client,
              getDataViews: async () => dataViewsService,
              share,
              getAsyncSearchClient: strategy => {
                const client = data.search.asScoped(request);
                return (0, _lib.wrapAsyncSearchClient)({
                  rule: {
                    name: rule.name,
                    id: rule.id,
                    alertTypeId: rule.ruleTypeId,
                    spaceId
                  },
                  logger,
                  strategy,
                  client,
                  abortController
                });
              }
            },
            spaceId,
            startedAt: startedAt.toDate(),
            startedAtOverridden: true,
            state: statePreview,
            logger,
            flappingSettings: _common.DISABLE_FLAPPING_SETTINGS,
            getTimeRange: () => {
              const date = startedAt.toISOString();
              return {
                dateStart: date,
                dateEnd: date
              };
            },
            isServerless,
            ruleExecutionTimeout: `${PREVIEW_TIMEOUT_SECONDS}s`
          }));
          const errors = loggedStatusChanges.filter(item => item.newStatus === _rule_monitoring.RuleExecutionStatusEnum.failed).map(item => {
            var _item$message;
            return (_item$message = item.message) !== null && _item$message !== void 0 ? _item$message : 'Unknown Error';
          });
          const warnings = loggedStatusChanges.filter(item => item.newStatus === _rule_monitoring.RuleExecutionStatusEnum['partial failure']).map(item => {
            var _item$message2;
            return (_item$message2 = item.message) !== null && _item$message2 !== void 0 ? _item$message2 : 'Unknown Warning';
          });
          logs.push({
            errors,
            warnings,
            startedAt: startedAt.toDate().toISOString(),
            duration: (0, _moment.default)().diff(invocationStartTime, 'milliseconds'),
            ...(loggedRequests ? {
              requests: loggedRequests
            } : {})
          });
          loggedStatusChanges.length = 0;
          if (errors.length) {
            break;
          }
          previousStartedAt = startedAt.toDate();
          startedAt.add((0, _utils2.parseInterval)(internalRule.schedule.interval));
          invocationCount--;
        }
      };
      switch (previewRuleParams.type) {
        case 'query':
          const queryAlertType = (0, _rule_types.createQueryAlertType)({
            id: _securitysolutionRules.QUERY_RULE_TYPE_ID,
            name: 'Custom Query Rule'
          });
          await runExecutors(queryAlertType, previewRuleParams);
          break;
        case 'saved_query':
          const savedQueryAlertType = (0, _rule_types.createQueryAlertType)({
            id: _securitysolutionRules.SAVED_QUERY_RULE_TYPE_ID,
            name: 'Saved Query Rule'
          });
          await runExecutors(savedQueryAlertType, previewRuleParams);
          break;
        case 'threshold':
          const thresholdAlertType = (0, _rule_types.createThresholdAlertType)();
          await runExecutors(thresholdAlertType, previewRuleParams);
          break;
        case 'threat_match':
          const threatMatchAlertType = (0, _rule_types.createIndicatorMatchAlertType)();
          await runExecutors(threatMatchAlertType, previewRuleParams);
          break;
        case 'eql':
          const eqlAlertType = (0, _rule_types.createEqlAlertType)();
          await runExecutors(eqlAlertType, previewRuleParams);
          break;
        case 'esql':
          if (config.experimentalFeatures.esqlRulesDisabled) {
            throw Error('ES|QL rule type is not supported');
          }
          const esqlAlertType = (0, _rule_types.createEsqlAlertType)();
          await runExecutors(esqlAlertType, previewRuleParams);
          break;
        case 'machine_learning':
          const mlAlertType = (0, _rule_types.createMlAlertType)(ml);
          await runExecutors(mlAlertType, previewRuleParams);
          break;
        case 'new_terms':
          const newTermsAlertType = (0, _rule_types.createNewTermsAlertType)();
          await runExecutors(newTermsAlertType, previewRuleParams);
          break;
        default:
          (0, _utility_types.assertUnreachable)(previewRuleParams);
      }

      // Refreshes alias to ensure index is able to be read before returning
      await coreContext.elasticsearch.client.asInternalUser.indices.refresh({
        index: previewRuleDataClient.indexNameWithNamespace(spaceId)
      }, {
        ignore: [404]
      });
      return response.ok({
        body: {
          previewId,
          logs,
          isAborted
        }
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: {
          errors: [error.message]
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.previewRulesRoute = previewRulesRoute;