"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRule = void 0;
var _convert_alerting_rule_to_rule_response = require("../converters/convert_alerting_rule_to_rule_response");
var _convert_rule_response_to_alerting_rule = require("../converters/convert_rule_response_to_alerting_rule");
var _apply_rule_update = require("../mergers/apply_rule_update");
var _utils = require("../utils");
var _create_rule = require("./create_rule");
var _get_rule_by_rule_id = require("./get_rule_by_rule_id");
var _errors = require("../../import/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const importRule = async ({
  actionsClient,
  rulesClient,
  importRulePayload,
  prebuiltRuleAssetClient,
  mlAuthz
}) => {
  const {
    ruleToImport,
    overwriteRules,
    overrideFields,
    allowMissingConnectorSecrets
  } = importRulePayload;
  // For backwards compatibility, immutable is false by default
  const rule = {
    ...ruleToImport,
    immutable: false,
    ...overrideFields
  };
  await (0, _utils.validateMlAuth)(mlAuthz, ruleToImport.type);
  const existingRule = await (0, _get_rule_by_rule_id.getRuleByRuleId)({
    rulesClient,
    ruleId: rule.rule_id
  });
  if (existingRule && !overwriteRules) {
    throw (0, _errors.createRuleImportErrorObject)({
      ruleId: existingRule.rule_id,
      type: 'conflict',
      message: 'Rule with this rule_id already exists'
    });
  }
  if (existingRule && overwriteRules) {
    let ruleWithUpdates = await (0, _apply_rule_update.applyRuleUpdate)({
      prebuiltRuleAssetClient,
      existingRule,
      ruleUpdate: rule
    });
    // applyRuleUpdate prefers the existing rule's values for `rule_source` and `immutable`, but we want to use the importing rule's calculated values
    ruleWithUpdates = {
      ...ruleWithUpdates,
      ...overrideFields
    };
    const updatedRule = await rulesClient.update({
      id: existingRule.id,
      data: (0, _convert_rule_response_to_alerting_rule.convertRuleResponseToAlertingRule)(ruleWithUpdates, actionsClient)
    });

    // We strip `enabled` from the rule object to use in the rules client and need to enable it separately if user has enabled the updated rule
    const {
      enabled
    } = await (0, _utils.toggleRuleEnabledOnUpdate)(rulesClient, existingRule, ruleWithUpdates);
    return (0, _convert_alerting_rule_to_rule_response.convertAlertingRuleToRuleResponse)({
      ...updatedRule,
      enabled
    });
  }

  /* Rule does not exist, so we'll create it */
  return (0, _create_rule.createRule)({
    actionsClient,
    rulesClient,
    mlAuthz,
    rule,
    allowMissingConnectorSecrets
  });
};
exports.importRule = importRule;