"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performRuleUpgradeHandler = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _utils = require("../../../../../../common/detection_engine/rule_management/utils");
var _prebuilt_rules = require("../../../../../../common/api/detection_engine/prebuilt_rules");
var _utils2 = require("../../../routes/utils");
var _convert_prebuilt_rule_asset_to_rule_response = require("../../../rule_management/logic/detection_rules_client/converters/convert_prebuilt_rule_asset_to_rule_response");
var _aggregate_prebuilt_rule_errors = require("../../logic/aggregate_prebuilt_rule_errors");
var _perform_timelines_installation = require("../../logic/perform_timelines_installation");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _upgrade_prebuilt_rules = require("../../logic/rule_objects/upgrade_prebuilt_rules");
var _create_upgradeable_rules_payload = require("./create_upgradeable_rules_payload");
var _validate_perform_rule_upgrade_request = require("./validate_perform_rule_upgrade_request");
var _zip_rule_versions = require("../../logic/rule_versions/zip_rule_versions");
var _calculate_rule_diff = require("../../logic/diff/calculate_rule_diff");
var _utils3 = require("../../logic/utils");
var _update_rule_telemetry = require("./update_rule_telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const performRuleUpgradeHandler = async (context, request, response, logger) => {
  const siemResponse = (0, _utils2.buildSiemResponse)(response);
  try {
    const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
    const soClient = ctx.core.savedObjects.client;
    const rulesClient = await ctx.alerting.getRulesClient();
    const detectionRulesClient = ctx.securitySolution.getDetectionRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
    const mlAuthz = ctx.securitySolution.getMlAuthz();
    const analyticsService = ctx.securitySolution.getAnalytics();
    const {
      isRulesCustomizationEnabled
    } = detectionRulesClient.getRuleCustomizationStatus();
    const defaultPickVersion = isRulesCustomizationEnabled ? _prebuilt_rules.PickVersionValuesEnum.MERGED : _prebuilt_rules.PickVersionValuesEnum.TARGET;
    (0, _validate_perform_rule_upgrade_request.validatePerformRuleUpgradeRequest)({
      isRulesCustomizationEnabled,
      payload: request.body,
      defaultPickVersion
    });
    const {
      mode,
      dry_run: isDryRun,
      on_conflict: onConflict
    } = request.body;
    const filter = mode === _prebuilt_rules.ModeEnum.ALL_RULES ? request.body.filter : undefined;
    const skippedRules = [];
    const updatedRules = [];
    const ruleErrors = [];
    const allErrors = [];
    const ruleUpgradeContextsMap = new Map();
    const ruleUpgradeQueue = [];
    if (mode === _prebuilt_rules.ModeEnum.ALL_RULES) {
      const allLatestVersions = await ruleAssetsClient.fetchLatestVersions();
      const latestVersionsMap = new Map(allLatestVersions.map(version => [version.rule_id, version]));
      const allCurrentVersions = await ruleObjectsClient.fetchInstalledRuleVersions({
        filter
      });
      const upgradableRules = await (0, _utils3.getPossibleUpgrades)(allCurrentVersions, latestVersionsMap, mlAuthz);
      ruleUpgradeQueue.push(...upgradableRules);
    } else if (mode === _prebuilt_rules.ModeEnum.SPECIFIC_RULES) {
      ruleUpgradeQueue.push(...request.body.rules);
    }
    const BATCH_SIZE = 100;
    while (ruleUpgradeQueue.length > 0) {
      const targetRulesForUpgrade = ruleUpgradeQueue.splice(0, BATCH_SIZE);
      const [currentRules, latestRules] = await Promise.all([ruleObjectsClient.fetchInstalledRulesByIds({
        ruleIds: targetRulesForUpgrade.map(({
          rule_id: ruleId
        }) => ruleId)
      }), ruleAssetsClient.fetchAssetsByVersion(targetRulesForUpgrade)]);
      const baseRules = await ruleAssetsClient.fetchAssetsByVersion(currentRules);
      const ruleVersionsMap = (0, _zip_rule_versions.zipRuleVersions)(currentRules, baseRules, latestRules);
      const upgradeableRules = [];
      targetRulesForUpgrade.forEach(targetRule => {
        const ruleVersions = ruleVersionsMap.get(targetRule.rule_id);
        const currentVersion = ruleVersions === null || ruleVersions === void 0 ? void 0 : ruleVersions.current;
        const baseVersion = ruleVersions === null || ruleVersions === void 0 ? void 0 : ruleVersions.base;
        const targetVersion = ruleVersions === null || ruleVersions === void 0 ? void 0 : ruleVersions.target;

        // Check that the requested rule was found
        if (!currentVersion) {
          ruleErrors.push({
            error: new Error(`Rule with rule_id "${targetRule.rule_id}" and version "${targetRule.version}" not found`),
            item: targetRule
          });
          return;
        }

        // Check that the requested rule is upgradeable
        if (!targetVersion || targetVersion.version <= currentVersion.version) {
          skippedRules.push({
            rule_id: targetRule.rule_id,
            reason: _prebuilt_rules.SkipRuleUpgradeReasonEnum.RULE_UP_TO_DATE
          });
          return;
        }

        // Check that rule revisions match (no update slipped in since the user reviewed the list)
        if (targetRule.revision != null && targetRule.revision !== currentVersion.revision) {
          ruleErrors.push({
            error: new Error(`Revision mismatch for rule_id ${targetRule.rule_id}: expected ${currentVersion.revision}, got ${targetRule.revision}`),
            item: targetRule
          });
          return;
        }

        // Check there's no conflicts
        if (onConflict === _prebuilt_rules.UpgradeConflictResolutionEnum.SKIP) {
          const ruleUpgradeSpecifier = request.body.mode === _prebuilt_rules.ModeEnum.SPECIFIC_RULES ? request.body.rules.find(x => x.rule_id === targetRule.rule_id) : undefined;
          const {
            ruleDiff
          } = (0, _calculate_rule_diff.calculateRuleDiff)(ruleVersions);
          const conflict = getRuleUpgradeConflictState(ruleDiff, ruleUpgradeSpecifier);
          if (conflict !== _prebuilt_rules.ThreeWayDiffConflict.NONE) {
            skippedRules.push({
              rule_id: targetRule.rule_id,
              reason: _prebuilt_rules.SkipRuleUpgradeReasonEnum.CONFLICT,
              conflict
            });
            ruleUpgradeContextsMap.set(targetRule.rule_id, {
              ruleId: targetRule.rule_id,
              ruleName: currentVersion.name,
              hasBaseVersion: !!baseVersion,
              isCustomized: (0, _utils.isRuleCustomized)(currentVersion),
              fieldsDiff: ruleDiff.fields
            });
            return;
          }
        }

        // All checks passed, add to the list of rules to upgrade
        upgradeableRules.push({
          current: currentVersion,
          base: baseVersion,
          target: targetVersion
        });
      });
      const {
        modifiedPrebuiltRuleAssets,
        processingErrors,
        ruleUpgradeContexts
      } = (0, _create_upgradeable_rules_payload.createModifiedPrebuiltRuleAssets)({
        upgradeableRules,
        requestBody: request.body,
        defaultPickVersion
      });
      ruleErrors.push(...processingErrors);
      ruleUpgradeContexts.forEach(ruleUpgradeContext => {
        ruleUpgradeContextsMap.set(ruleUpgradeContext.ruleId, ruleUpgradeContext);
      });
      if (isDryRun) {
        updatedRules.push(...modifiedPrebuiltRuleAssets.map(rule => (0, _convert_prebuilt_rule_asset_to_rule_response.convertPrebuiltRuleAssetToRuleResponse)(rule)));
      } else {
        const {
          results: upgradeResults,
          errors: installationErrors
        } = await (0, _upgrade_prebuilt_rules.upgradePrebuiltRules)(detectionRulesClient, modifiedPrebuiltRuleAssets, logger);
        ruleErrors.push(...installationErrors);
        updatedRules.push(...upgradeResults.map(({
          result
        }) => result));
      }
    }
    allErrors.push(...(0, _aggregate_prebuilt_rule_errors.aggregatePrebuiltRuleErrors)(ruleErrors));
    if (!isDryRun) {
      const {
        error: timelineInstallationError
      } = await (0, _perform_timelines_installation.performTimelinesInstallation)(ctx.securitySolution);
      if (timelineInstallationError) {
        allErrors.push({
          message: timelineInstallationError,
          rules: []
        });
      }
      (0, _update_rule_telemetry.sendRuleUpdateTelemetryEvents)(analyticsService, ruleUpgradeContextsMap, updatedRules, ruleErrors, skippedRules, logger);
      if (mode === _prebuilt_rules.ModeEnum.ALL_RULES) {
        (0, _update_rule_telemetry.sendRuleBulkUpgradeTelemetryEvent)(analyticsService, ruleUpgradeContextsMap, updatedRules, ruleErrors, skippedRules, logger);
      }
    }
    const body = {
      summary: {
        total: updatedRules.length + skippedRules.length + ruleErrors.length,
        skipped: skippedRules.length,
        succeeded: updatedRules.length,
        failed: ruleErrors.length
      },
      results: {
        updated: updatedRules,
        skipped: skippedRules
      },
      errors: allErrors
    };
    return response.ok({
      body
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};
exports.performRuleUpgradeHandler = performRuleUpgradeHandler;
function getRuleUpgradeConflictState(ruleDiff, ruleUpgradeSpecifier) {
  if (ruleDiff.num_fields_with_conflicts === 0) {
    return _prebuilt_rules.ThreeWayDiffConflict.NONE;
  }
  if (!ruleUpgradeSpecifier) {
    return ruleDiff.num_fields_with_non_solvable_conflicts > 0 ? _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE : _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE;
  }
  let result = _prebuilt_rules.ThreeWayDiffConflict.NONE;

  // filter out resolved fields
  for (const [fieldName, fieldThreeWayDiff] of Object.entries(ruleDiff.fields)) {
    var _ruleUpgradeSpecifier, _ruleUpgradeSpecifier2;
    const hasResolvedValue = ((_ruleUpgradeSpecifier = ruleUpgradeSpecifier.fields) === null || _ruleUpgradeSpecifier === void 0 ? void 0 : (_ruleUpgradeSpecifier2 = _ruleUpgradeSpecifier[fieldName]) === null || _ruleUpgradeSpecifier2 === void 0 ? void 0 : _ruleUpgradeSpecifier2.pick_version) === 'RESOLVED';
    if (fieldThreeWayDiff.conflict === _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE && !hasResolvedValue) {
      return _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE;
    } else if (fieldThreeWayDiff.conflict === _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE && !hasResolvedValue) {
      result = _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE;
    }
  }
  return result;
}