"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.wrapErrorIfNeeded = exports.catchAndWrapError = void 0;
var _server = require("@kbn/fleet-plugin/server");
var _errors = require("@kbn/fleet-plugin/server/errors");
var _errors2 = require("../errors");
var _errors3 = require("../../../common/endpoint/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Will wrap the given Error with `EndpointError`, which will help getting a good picture of where in
 * our code the error originated (better stack trace).
 */
const wrapErrorIfNeeded = (error, messagePrefix) => {
  if (error instanceof _errors3.EndpointError) {
    return error;
  }
  const message = `${messagePrefix ? `${messagePrefix}: ` : ''}${error.message}`;

  // Check for known "Not Found" errors and wrap them with our own `NotFoundError`, which will enable
  // the correct HTTP status code to be used if it is thrown during processing of an API route
  if (error instanceof _server.AgentNotFoundError || error instanceof _errors.AgentPolicyNotFoundError || error instanceof _errors.PackagePolicyNotFoundError) {
    return new _errors2.NotFoundError(message, error);
  }
  return new _errors3.EndpointError(message, error);
};
exports.wrapErrorIfNeeded = wrapErrorIfNeeded;
/**
 * used as the callback to `Promise#catch()` to ensure errors
 * (especially those from kibana/elasticsearch clients) are wrapped
 *
 * @param error
 */
const catchAndWrapError = error => Promise.reject(wrapErrorIfNeeded(error));
exports.catchAndWrapError = catchAndWrapError;
catchAndWrapError.withMessage = message => {
  return err => Promise.reject(wrapErrorIfNeeded(err, message));
};