"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsServiceBase = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _constants = require("@kbn/elastic-assistant/impl/assistant_context/constants");
var _constants2 = require("../../../../common/siem_migrations/constants");
var _experimental_features_service = require("../../../common/experimental_features_service");
var _use_license = require("../../../common/hooks/use_license");
var _capabilities = require("./capabilities");
var _storage = require("./storage");
var i18n = _interopRequireWildcard(require("./translations"));
var _constants3 = require("../constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// use the default assistant namespace since it's the only one we use
const NAMESPACE_TRACE_OPTIONS_SESSION_STORAGE_KEY = `${_constants.DEFAULT_ASSISTANT_NAMESPACE}.${_constants.TRACE_OPTIONS_SESSION_STORAGE_KEY}`;
class SiemMigrationsServiceBase {
  constructor(core, plugins) {
    (0, _defineProperty2.default)(this, "latestStats$", void 0);
    (0, _defineProperty2.default)(this, "isPolling", false);
    (0, _defineProperty2.default)(this, "connectorIdStorage", void 0);
    (0, _defineProperty2.default)(this, "traceOptionsStorage", void 0);
    this.core = core;
    this.plugins = plugins;
    this.connectorIdStorage = new _storage.MigrationsStorage('connectorId');
    this.traceOptionsStorage = new _storage.MigrationsStorage('traceOptions', {
      customKey: NAMESPACE_TRACE_OPTIONS_SESSION_STORAGE_KEY,
      storageType: 'session'
    });
    this.latestStats$ = new _rxjs.BehaviorSubject(null);
    this.plugins.spaces.getActiveSpace().then(space => {
      this.connectorIdStorage.setSpaceId(space.id);
      this.startPolling();
    });
  }

  /** Returns the latest stats observable, which is updated every time the stats are fetched */
  getLatestStats$() {
    return this.latestStats$.asObservable().pipe((0, _rxjs.distinctUntilChanged)(_lodash.isEqual));
  }

  /** Returns any missing capabilities for the user to use this feature */
  getMissingCapabilities(level) {
    const getMissingCapabilities = (0, _capabilities.getMissingCapabilitiesChecker)();
    return getMissingCapabilities(this.core.application.capabilities, level);
  }

  /** Checks if the user has any missing capabilities for this feature */
  hasMissingCapabilities(level) {
    return this.getMissingCapabilities(level).length > 0;
  }

  /** Checks if the service is available based on the `license`, `capabilities` and `experimentalFeatures` */
  isAvailable() {
    return !_experimental_features_service.ExperimentalFeaturesService.get().siemMigrationsDisabled && _use_license.licenseService.isEnterprise() && !this.hasMissingCapabilities('minimum');
  }

  /** Starts polling the migrations stats if not already polling and if the feature is available to the user */
  startPolling() {
    if (this.isPolling || !this.isAvailable()) {
      return;
    }
    this.isPolling = true;
    this.startTaskStatsPolling().catch(e => {
      this.core.notifications.toasts.addError(e, {
        title: i18n.POLLING_ERROR
      });
    }).finally(() => {
      this.isPolling = false;
    });
  }

  /** Gets the migrations stats, retrying on network errors or 503 status */
  async getMigrationsStats(params = {}) {
    const allStats = await this.getMigrationsStatsWithRetry(params);
    this.latestStats$.next(allStats); // Keep the latest stats observable in sync
    return allStats;
  }
  sleep(seconds) {
    return new Promise(resolve => setTimeout(resolve, seconds * 1000));
  }

  /** Polls the migration task stats until the finish condition is met or the timeout is reached. */
  async migrationTaskPollingUntil(migrationId, finishCondition, {
    sleepSecs = 1,
    timeoutSecs = 60
  } = {}) {
    const timeoutId = setTimeout(() => {
      throw new Error('Migration task polling timed out');
    }, timeoutSecs * 1000);
    let retry = true;
    do {
      const stats = await this.fetchMigrationStats({
        migrationId
      });
      if (finishCondition(stats)) {
        clearTimeout(timeoutId);
        retry = false;
      } else {
        await this.sleep(sleepSecs);
      }
    } while (retry);
    // updates the latest stats observable for all migrations to make sure they are in sync
    await this.getMigrationsStats();
  }

  /** Retries the API call to get migrations stats in case of network errors or 503 status */
  async getMigrationsStatsWithRetry(params = {}, sleepSecs) {
    if (sleepSecs) {
      await this.sleep(sleepSecs);
    }
    return this.fetchMigrationsStatsAll(params).catch(e => {
      var _e$response;
      // Retry only on network errors (no status) and 503 (Service Unavailable), otherwise throw
      const status = ((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) || e.status;
      if (status && status !== 503) {
        throw e;
      }
      const nextSleepSecs = sleepSecs ? sleepSecs * 2 : 1; // Exponential backoff
      if (nextSleepSecs > 60) {
        // Wait for a minutes max (two minutes total) for the API to be available again
        throw e;
      }
      return this.getMigrationsStatsWithRetry(params, nextSleepSecs);
    });
  }

  /** Starts polling the migrations stats and handles the notifications for finished migrations */
  async startTaskStatsPolling() {
    let pendingMigrationIds = [];
    do {
      const results = await this.getMigrationsStats();
      if (pendingMigrationIds.length > 0) {
        // send notifications for finished migrations
        pendingMigrationIds.forEach(pendingMigrationId => {
          const migrationStats = results.find(item => item.id === pendingMigrationId);
          if ((migrationStats === null || migrationStats === void 0 ? void 0 : migrationStats.status) === _constants2.SiemMigrationTaskStatus.FINISHED) {
            this.sendFinishedMigrationNotification(migrationStats);
          }
        });
      }

      // reprocess pending migrations
      pendingMigrationIds = [];
      for (const result of results) {
        var _result$last_executio;
        if (result.status === _constants2.SiemMigrationTaskStatus.RUNNING) {
          pendingMigrationIds.push(result.id);
        }

        // automatically resume interrupted migrations when the proper conditions are met
        if (result.status === _constants2.SiemMigrationTaskStatus.INTERRUPTED && !((_result$last_executio = result.last_execution) !== null && _result$last_executio !== void 0 && _result$last_executio.error)) {
          var _result$last_executio2, _result$last_executio3;
          const connectorId = (_result$last_executio2 = (_result$last_executio3 = result.last_execution) === null || _result$last_executio3 === void 0 ? void 0 : _result$last_executio3.connector_id) !== null && _result$last_executio2 !== void 0 ? _result$last_executio2 : this.connectorIdStorage.get();
          if (connectorId && !this.hasMissingCapabilities('all')) {
            await this.startMigrationFromStats(connectorId, result);
            pendingMigrationIds.push(result.id);
          }
        }
      }

      // Do not wait if there are no more pending migrations
      if (pendingMigrationIds.length > 0) {
        await this.sleep(_constants3.TASK_STATS_POLLING_SLEEP_SECONDS);
      }
    } while (pendingMigrationIds.length > 0);
  }
}
exports.SiemMigrationsServiceBase = SiemMigrationsServiceBase;