"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIntegrationLastAlertIngested = void 0;
var _react = require("react");
var _reactQuery = require("@kbn/react-query");
var _esqlUtils = require("@kbn/esql-utils");
var _esql = require("../../../common/utils/esql");
var _kibana = require("../../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIELD = 'event.ingested';
/**
 * Hook that fetches the last alert ingested time for a specific integration.
 * We use the index pattern `logs-{integrationName}.alert-default` to query.
 */
const useIntegrationLastAlertIngested = ({
  integrationName
}) => {
  const {
    data
  } = (0, _kibana.useKibana)().services;

  // ESQL query to get the last alert ingested in the index
  // We only keep the event.ingested field as it contains the time we want to display on the Integration card.
  const query = (0, _react.useMemo)(() => `FROM ${`logs-${integrationName}.alert-default`}
    | WHERE event.kind == "alert" OR event.kind == "event"
    | SORT ${FIELD} DESC
    | KEEP ${FIELD}
    | LIMIT 1`, [integrationName]);
  const {
    isLoading,
    data: result,
    refetch
  } = (0, _reactQuery.useQuery)([integrationName], async ({
    signal
  }) => (0, _esqlUtils.getESQLResults)({
    esqlQuery: query,
    search: data.search.search,
    signal
  }), {
    refetchOnWindowFocus: false,
    keepPreviousData: true
  });
  const lastAlertIngested = (0, _react.useMemo)(() => {
    const records = (0, _esql.esqlResponseToRecords)(result === null || result === void 0 ? void 0 : result.response);
    return records.length > 0 ? records[0][FIELD] : null;
  }, [result === null || result === void 0 ? void 0 : result.response]);
  return (0, _react.useMemo)(() => ({
    isLoading,
    lastAlertIngested,
    refetch
  }), [isLoading, lastAlertIngested, refetch]);
};
exports.useIntegrationLastAlertIngested = useIntegrationLastAlertIngested;