"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateIntegrationDetails = void 0;
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Given an array of integrations and an array of all known integrations this will return an
 * array of integrations augmented with details like targetVersion, and `version_satisfied`
 * has.
 */
const calculateIntegrationDetails = (relatedIntegrations, knownIntegrations) => {
  const integrationMatches = findIntegrationMatches(relatedIntegrations, knownIntegrations);
  const integrationDetails = integrationMatches.map(integration => createIntegrationDetails(integration));
  return integrationDetails.sort((a, b) => a.integrationTitle.localeCompare(b.integrationTitle));
};
exports.calculateIntegrationDetails = calculateIntegrationDetails;
const findIntegrationMatches = (relatedIntegrations, integrations) => {
  const integrationsMap = new Map((integrations !== null && integrations !== void 0 ? integrations : []).map(integration => {
    var _integration$integrat;
    return [`${integration.package_name}${(_integration$integrat = integration.integration_name) !== null && _integration$integrat !== void 0 ? _integration$integrat : ''}`, integration];
  }));
  return relatedIntegrations.map(ri => {
    var _ri$integration;
    const key = `${ri.package}${(_ri$integration = ri.integration) !== null && _ri$integration !== void 0 ? _ri$integration : ''}`;
    const matchIntegration = integrationsMap.get(key);
    if (!matchIntegration) {
      return {
        related: ri,
        isLoaded: false
      };
    }
    return {
      related: ri,
      found: matchIntegration,
      isLoaded: true
    };
  });
};
const createIntegrationDetails = integration => {
  var _related$integration, _found$integration_ti, _found$installed_pack;
  const {
    related,
    found,
    isLoaded
  } = integration;
  const packageName = related.package;
  const integrationName = (_related$integration = related.integration) !== null && _related$integration !== void 0 ? _related$integration : null;
  const requiredVersion = related.version;

  // We don't know whether the integration is installed or not.
  if (!isLoaded) {
    const integrationTitle = getCapitalizedTitle(packageName, integrationName);
    const targetVersion = getMinimumConcreteVersionMatchingSemver(requiredVersion);
    const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
    return {
      packageName,
      integrationName,
      integrationTitle,
      requiredVersion,
      targetVersion,
      targetUrl,
      installationStatus: {
        isKnown: false
      }
    };
  }
  if (!found) {
    const integrationTitle = getCapitalizedTitle(packageName, integrationName);
    const targetVersion = getMinimumConcreteVersionMatchingSemver(requiredVersion);
    const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
    return {
      packageName,
      integrationName,
      integrationTitle,
      requiredVersion,
      targetVersion,
      targetUrl,
      installationStatus: {
        isKnown: true,
        isInstalled: false,
        isEnabled: false,
        isVersionMismatch: false,
        installedVersion: ''
      }
    };
  }
  const integrationTitle = (_found$integration_ti = found.integration_title) !== null && _found$integration_ti !== void 0 ? _found$integration_ti : found.package_title;
  // Version check e.g. installed version `1.2.3` satisfies required version `~1.2.1`
  const installedVersion = (_found$installed_pack = found.installed_package_version) !== null && _found$installed_pack !== void 0 ? _found$installed_pack : '';
  const isVersionSatisfied = installedVersion ? _semver.default.satisfies(installedVersion, requiredVersion, {
    includePrerelease: true
  }) : true;
  const targetVersion = installedVersion && isVersionSatisfied ? installedVersion : getMinimumConcreteVersionMatchingSemver(requiredVersion);
  const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
  return {
    packageName,
    integrationName,
    integrationTitle,
    requiredVersion,
    targetVersion,
    targetUrl,
    installationStatus: {
      isKnown: true,
      isInstalled: found.is_installed,
      isEnabled: found.is_enabled,
      isVersionMismatch: !isVersionSatisfied,
      installedVersion
    }
  };
};
const getCapitalizedTitle = (packageName, integrationName) => {
  return integrationName == null ? `${(0, _lodash.capitalize)(packageName)}` : `${(0, _lodash.capitalize)(packageName)} ${(0, _lodash.capitalize)(integrationName)}`;
};
const getMinimumConcreteVersionMatchingSemver = semverString => {
  var _semver$valid;
  return (_semver$valid = _semver.default.valid(_semver.default.coerce(semverString))) !== null && _semver$valid !== void 0 ? _semver$valid : '';
};
const buildTargetUrl = (packageName, integrationName, targetVersion) => {
  const packageSegment = targetVersion ? `${packageName}-${targetVersion}` : packageName;
  const query = integrationName ? `?integration=${integrationName}` : '';
  return `app/integrations/detail/${packageSegment}/overview${query}`;
};