"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useNavigation = exports.useNavigateTo = exports.useGetAppUrl = void 0;
var _react = require("react");
var _constants = require("./constants");
var _context = require("./context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The `useGetAppUrl` function returns a full URL to the provided page path by using
 * kibana's `getUrlForApp()`
 */
const useGetAppUrl = () => {
  const {
    getUrlForApp
  } = (0, _context.useNavigationContext)().application;
  const getAppUrl = (0, _react.useCallback)(({
    appId = _constants.SECURITY_UI_APP_ID,
    ...options
  }) => getUrlForApp(appId, options), [getUrlForApp]);
  return {
    getAppUrl
  };
};
exports.useGetAppUrl = useGetAppUrl;
/**
 * The `navigateTo` function navigates to any app using kibana's `navigateToApp()`.
 * When the `{ url: string }` parameter is passed it will navigate using `navigateToUrl()`.
 */
const useNavigateTo = () => {
  const {
    navigateToApp,
    navigateToUrl
  } = (0, _context.useNavigationContext)().application;
  const navigateTo = (0, _react.useCallback)(({
    url,
    appId = _constants.SECURITY_UI_APP_ID,
    restoreScroll,
    ...options
  }) => {
    if (restoreScroll) {
      addScrollRestoration();
    }
    if (url) {
      navigateToUrl(url);
    } else {
      navigateToApp(appId, options);
    }
  }, [navigateToApp, navigateToUrl]);
  return {
    navigateTo
  };
};

/**
 * Expects the browser scroll reset event to be fired after the navigation,
 * then restores the previous scroll position.
 */
exports.useNavigateTo = useNavigateTo;
const addScrollRestoration = () => {
  const scrollY = window.scrollY;
  const handler = () => window.scrollTo(0, scrollY);
  window.addEventListener('scroll', handler, {
    once: true
  });
};

/**
 * Returns `navigateTo` and `getAppUrl` navigation hooks
 */
const useNavigation = () => {
  const {
    navigateTo
  } = useNavigateTo();
  const {
    getAppUrl
  } = useGetAppUrl();
  return {
    navigateTo,
    getAppUrl
  };
};
exports.useNavigation = useNavigation;