"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addSubFeatureReplacements = exports.addAllSubFeatureReplacements = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Adds the replacedBy entries to the subFeature's privileges.
 * It does not mutate the original subFeature.
 * @param subFeature - The subFeature to add replacements to
 * @param replacements - The replacements to add
 * @returns A new subFeature with the replacements added
 */
const addSubFeatureReplacements = (subFeature, replacements) => {
  if (!replacements.length) {
    return subFeature;
  }
  const subFeatureWithReplacement = (0, _lodash.cloneDeep)(subFeature);
  subFeatureWithReplacement.privilegeGroups.forEach(privilegeGroup => {
    privilegeGroup.privileges.forEach(privilege => {
      privilege.replacedBy ??= [];
      for (const replacement of replacements) {
        var _replacement$addition, _replacement$addition2;
        const privileges = !replacement.removeOriginalPrivileges ? [privilege.id] : [];
        privileges.push(...((_replacement$addition = (_replacement$addition2 = replacement.additionalPrivileges) === null || _replacement$addition2 === void 0 ? void 0 : _replacement$addition2[privilege.id]) !== null && _replacement$addition !== void 0 ? _replacement$addition : []));
        privilege.replacedBy.push({
          feature: replacement.feature,
          privileges
        });
      }
    });
  });
  return subFeatureWithReplacement;
};

/**
 * Adds the replacements to all sub-features in the provided subFeaturesMap.
 * It does not mutate the original subFeaturesMap.
 * @param subFeaturesMap - The subFeaturesMap to add replacements to
 * @param replacements - The replacements to add
 * @returns A new subFeaturesMap with the replacements added
 */
exports.addSubFeatureReplacements = addSubFeatureReplacements;
const addAllSubFeatureReplacements = (subFeaturesMap, replacements) => {
  if (!replacements.length) {
    return subFeaturesMap;
  }
  return new Map([...subFeaturesMap.entries()].map(([id, subFeature]) => {
    const subFeatureWithReplacement = addSubFeatureReplacements(subFeature, replacements);
    return [id, subFeatureWithReplacement];
  }));
};
exports.addAllSubFeatureReplacements = addAllSubFeatureReplacements;