"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.featureConfigMerger = exports.extendProductFeatureConfigs = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Custom merge function for product feature configs. To be used with `mergeWith`.
 * It merges arrays by removing duplicates by shallow comparison and extends other properties.
 * It does not mutate the original objects.
 * @param objValue - The value from the first object
 * @param srcValue - The value from the second object
 * @returns The merged value
 */
const featureConfigMerger = (objValue, srcValue) => {
  if (Array.isArray(objValue) && Array.isArray(srcValue)) {
    return (0, _lodash.uniq)(objValue.concat(srcValue));
  }
  return undefined; // Use default merge behavior for other types
};

/**
 * Extends multiple ProductFeaturesConfig objects into a single one.
 * It merges arrays by removing duplicates and keeps the rest of the properties as is.
 * It does not mutate the original objects.
 *
 * @param productFeatureConfigs - The product feature configs to merge
 * @returns A single extended ProductFeaturesConfig object
 */
exports.featureConfigMerger = featureConfigMerger;
const extendProductFeatureConfigs = (...productFeatureConfigs) => {
  return (0, _lodash.mergeWith)({}, ...productFeatureConfigs, featureConfigMerger);
};
exports.extendProductFeatureConfigs = extendProductFeatureConfigs;