"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateEsPrivilegeResponse = validateEsPrivilegeResponse;
var _configSchema = require("@kbn/config-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const baseResponseSchema = _configSchema.schema.object({
  username: _configSchema.schema.string(),
  has_all_requested: _configSchema.schema.boolean(),
  application: _configSchema.schema.object({}, {
    unknowns: 'allow'
  }),
  cluster: _configSchema.schema.object({}, {
    unknowns: 'allow'
  }),
  index: _configSchema.schema.object({}, {
    unknowns: 'allow'
  })
});

/**
 * Validates an Elasticsearch "Has privileges" response against the expected application, actions, and resources.
 *
 * Note: the `actions` and `resources` parameters must be unique string arrays; any duplicates will cause validation to fail.
 */
function validateEsPrivilegeResponse(response, application, actions, resources) {
  try {
    baseResponseSchema.validate(response);
    validateResponse(response, application, actions, resources);
  } catch (e) {
    throw new Error(`Invalid response received from Elasticsearch has_privilege endpoint. ${e}`);
  }
  return response;
}
const validateResponse = (response, applicationName, actionNames, resourceNames) => {
  var _response$application;
  const actualApplicationNames = Object.keys((_response$application = response.application) !== null && _response$application !== void 0 ? _response$application : {});
  if (actualApplicationNames.length !== 1) {
    throw new Error(`Expected one application but received ${actualApplicationNames.length}`);
  }
  if (actualApplicationNames[0] !== applicationName) {
    throw new Error(`Expected application to be ${applicationName} but received ${actualApplicationNames[0]}`);
  }
  const actualApplication = response.application[applicationName];
  const actualResourceNames = Object.keys(actualApplication).sort();
  if (resourceNames.length !== actualResourceNames.length || ![...resourceNames].sort().every((x, i) => x === actualResourceNames[i])) {
    throw new Error('Payload did not match expected resources');
  }
  const sortedActionNames = [...actionNames].sort();
  Object.values(actualApplication).forEach(actualResource => {
    const actualActionNames = Object.keys(actualResource).sort();
    if (actionNames.length !== actualActionNames.length || !sortedActionNames.every((x, i) => x === actualActionNames[i])) {
      throw new Error('Payload did not match expected actions');
    }
    Object.values(actualResource).forEach(actualActionValue => {
      if (typeof actualActionValue !== 'boolean') {
        throw new Error('Payload did not match expected action values');
      }
    });
  });
};