"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateRoleMappingForSave = validateRoleMappingForSave;
exports.validateRoleMappingName = validateRoleMappingName;
exports.validateRoleMappingRoleTemplates = validateRoleMappingRoleTemplates;
exports.validateRoleMappingRoles = validateRoleMappingRoles;
exports.validateRoleMappingRules = validateRoleMappingRules;
var _i18n = require("@kbn/i18n");
var _model = require("../../model");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateRoleMappingName({
  name
}) {
  if (!name) {
    return invalid(_i18n.i18n.translate('xpack.security.role_mappings.validation.invalidName', {
      defaultMessage: 'Name is required.'
    }));
  }
  return valid();
}
function validateRoleMappingRoles({
  roles
}) {
  if (roles && !roles.length) {
    return invalid(_i18n.i18n.translate('xpack.security.role_mappings.validation.invalidRoles', {
      defaultMessage: 'At least one role is required.'
    }));
  }
  return valid();
}
function validateRoleMappingRoleTemplates({
  role_templates: roleTemplates
}) {
  if (roleTemplates && !roleTemplates.length) {
    return invalid(_i18n.i18n.translate('xpack.security.role_mappings.validation.invalidRoleTemplates', {
      defaultMessage: 'At least one role template is required.'
    }));
  }
  return valid();
}
function validateRoleMappingRules({
  rules
}) {
  try {
    const {
      rules: parsedRules
    } = (0, _model.generateRulesFromRaw)(rules);
    if (!parsedRules) {
      return invalid(_i18n.i18n.translate('xpack.security.role_mappings.validation.invalidRoleRule', {
        defaultMessage: 'At least one rule is required.'
      }));
    }
  } catch (e) {
    return invalid(e.message);
  }
  return valid();
}
function validateRoleMappingForSave(roleMapping) {
  const {
    isInvalid: isNameInvalid,
    error: nameError
  } = validateRoleMappingName(roleMapping);
  const {
    isInvalid: areRolesInvalid,
    error: rolesError
  } = validateRoleMappingRoles(roleMapping);
  const {
    isInvalid: areRoleTemplatesInvalid,
    error: roleTemplatesError
  } = validateRoleMappingRoleTemplates(roleMapping);
  const {
    isInvalid: areRulesInvalid,
    error: rulesError
  } = validateRoleMappingRules(roleMapping);
  const canSave = !isNameInvalid && (!areRolesInvalid || !areRoleTemplatesInvalid) && !areRulesInvalid;
  if (canSave) {
    return valid();
  }
  return invalid(nameError || rulesError || rolesError || roleTemplatesError);
}
function valid() {
  return {
    isInvalid: false
  };
}
function invalid(error) {
  return {
    isInvalid: true,
    error
  };
}