"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseStreamPart = exports.formatStreamPart = void 0;
exports.readDataStream = readDataStream;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Maps the type of a stream part to its value type.
 */

const NEWLINE = '\n'.charCodeAt(0);
const concatChunks = (chunks, totalLength) => {
  const concatenatedChunks = new Uint8Array(totalLength);
  let offset = 0;
  for (const chunk of chunks) {
    concatenatedChunks.set(chunk, offset);
    offset += chunk.length;
  }
  chunks.length = 0;
  return concatenatedChunks;
};
async function* readDataStream(reader, {
  isAborted
} = {}) {
  const decoder = new TextDecoder();
  const chunks = [];
  let totalLength = 0;
  while (true) {
    const {
      value
    } = await reader.read();
    if (value) {
      chunks.push(value);
      totalLength += value.length;
      if (value[value.length - 1] !== NEWLINE) {
        continue;
      }
    }
    if (chunks.length === 0) {
      break;
    }
    const concatenatedChunks = concatChunks(chunks, totalLength);
    totalLength = 0;
    const streamParts = decoder.decode(concatenatedChunks, {
      stream: true
    }).split('\n').filter(line => line !== '').map(parseStreamPart);
    for (const streamPart of streamParts) {
      yield streamPart;
    }
    if (isAborted !== null && isAborted !== void 0 && isAborted()) {
      reader.cancel();
      break;
    }
  }
}
const createStreamPart = (code, name, parse) => {
  return {
    code,
    name,
    parse
  };
};
const textStreamPart = createStreamPart('0', 'text', value => {
  if (typeof value !== 'string') {
    throw new Error('"text" parts expect a string value.');
  }
  return {
    type: 'text',
    value
  };
});
const errorStreamPart = createStreamPart('3', 'error', value => {
  if (typeof value !== 'string') {
    throw new Error('"error" parts expect a string value.');
  }
  return {
    type: 'error',
    value
  };
});
const messageAnnotationsStreamPart = createStreamPart('8', 'message_annotations', value => {
  if (!Array.isArray(value)) {
    throw new Error('"message_annotations" parts expect an array value.');
  }
  return {
    type: 'message_annotations',
    value
  };
});
const bufferStreamPart = createStreamPart('10', 'buffer', value => {
  if (typeof value !== 'string') {
    throw new Error('"buffer" parts expect a string value.');
  }
  return {
    type: 'buffer',
    value
  };
});
const streamParts = [textStreamPart, errorStreamPart, bufferStreamPart, messageAnnotationsStreamPart];
const streamPartsByCode = streamParts.reduce((acc, part) => ({
  ...acc,
  [part.code]: part
}), {});
const validCodes = streamParts.map(part => part.code);
const parseStreamPart = line => {
  const firstSeparatorIndex = line.indexOf(':');
  if (firstSeparatorIndex === -1) {
    throw new Error('Failed to parse stream string. No separator found.');
  }
  const prefix = line.slice(0, firstSeparatorIndex);
  if (!validCodes.includes(prefix)) {
    throw new Error(`Failed to parse stream string. Invalid code ${prefix}.`);
  }
  const code = prefix;
  const textValue = line.slice(firstSeparatorIndex + 1);
  const jsonValue = JSON.parse(textValue);
  return streamPartsByCode[code].parse(jsonValue);
};
exports.parseStreamPart = parseStreamPart;
const formatStreamPart = (type, value) => {
  const streamPart = streamParts.find(part => part.name === type);
  if (!streamPart) {
    throw new Error(`Invalid stream part type: ${type}`);
  }
  return `${streamPart.code}:${JSON.stringify(value)}\n`;
};
exports.formatStreamPart = formatStreamPart;