"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createFetchESFunctions = createFetchESFunctions;
var _common = require("@kbn/observability-plugin/common");
var _profilingUtils = require("@kbn/profiling-utils");
var _percent_to_factor = require("../../utils/percent_to_factor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const targetSampleSize = 20000; // minimum number of samples to get statistically sound results

function createFetchESFunctions({
  createProfilingEsClient
}) {
  return async ({
    core,
    esClient,
    indices,
    stacktraceIdsField,
    query,
    aggregationFields,
    limit,
    totalSeconds
  }) => {
    const [co2PerKWH, datacenterPUE, pervCPUWattX86, pervCPUWattArm64, awsCostDiscountRate, costPervCPUPerHour, azureCostDiscountRate] = await Promise.all([core.uiSettings.client.get(_common.profilingCo2PerKWH), core.uiSettings.client.get(_common.profilingDatacenterPUE), core.uiSettings.client.get(_common.profilingPervCPUWattX86), core.uiSettings.client.get(_common.profilingPervCPUWattArm64), core.uiSettings.client.get(_common.profilingAWSCostDiscountRate), core.uiSettings.client.get(_common.profilingCostPervCPUPerHour), core.uiSettings.client.get(_common.profilingAzureCostDiscountRate), core.uiSettings.client.get(_common.profilingShowErrorFrames)]);
    const profilingEsClient = createProfilingEsClient({
      esClient
    });
    const esTopNFunctions = await profilingEsClient.topNFunctions({
      sampleSize: targetSampleSize,
      limit,
      query,
      indices,
      stacktraceIdsField,
      aggregationFields,
      co2PerKWH,
      datacenterPUE,
      pervCPUWattX86,
      pervCPUWattArm64,
      awsCostDiscountRate: (0, _percent_to_factor.percentToFactor)(awsCostDiscountRate),
      costPervCPUPerHour,
      azureCostDiscountRate: (0, _percent_to_factor.percentToFactor)(azureCostDiscountRate),
      durationSeconds: totalSeconds
    });
    return transformToKibanaTopNFunction(esTopNFunctions);
  };
}

/**
 * Transforms object returned by ES because we share a lot of components in the UI with the current data model
 * We must first align the ES api response type then remove this
 */
function transformToKibanaTopNFunction(esTopNFunctions) {
  return {
    TotalCount: esTopNFunctions.total_count,
    totalCPU: esTopNFunctions.total_count,
    selfCPU: esTopNFunctions.self_count,
    totalAnnualCO2Kgs: (0, _profilingUtils.convertTonsToKgs)(esTopNFunctions.self_annual_co2_tons),
    totalAnnualCostUSD: esTopNFunctions.self_annual_cost_usd,
    SamplingRate: 1,
    TopN: esTopNFunctions.topn.map(item => {
      return {
        Id: item.id,
        Rank: item.rank,
        CountExclusive: item.self_count,
        CountInclusive: item.total_count,
        selfAnnualCO2kgs: (0, _profilingUtils.convertTonsToKgs)(item.self_annual_co2_tons),
        selfAnnualCostUSD: item.self_annual_costs_usd,
        totalAnnualCO2kgs: (0, _profilingUtils.convertTonsToKgs)(item.total_annual_co2_tons),
        totalAnnualCostUSD: item.total_annual_costs_usd,
        subGroups: item.sub_groups,
        Frame: {
          AddressOrLine: item.frame.address_or_line,
          ExeFileName: item.frame.executable_file_name,
          FrameType: item.frame.frame_type,
          FunctionName: item.frame.function_name,
          Inline: item.frame.inline,
          SourceFilename: item.frame.file_name,
          SourceLine: item.frame.line_number,
          FileID: '',
          FrameID: '',
          FunctionOffset: 0
        }
      };
    })
  };
}