"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toolsToLangchain = exports.toolToLangchain = exports.toolIdentifierFromToolCall = exports.sanitizeToolId = exports.createToolIdMappings = void 0;
var _lodash = require("lodash");
var _zod = require("@kbn/zod");
var _tools = require("@langchain/core/tools");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatServer = require("@kbn/onechat-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolsToLangchain = async ({
  request,
  tools,
  logger,
  sendEvent,
  addReasoningParam = true
}) => {
  const allTools = Array.isArray(tools) ? tools : await tools.list({
    request
  });
  const onechatToLangchainIdMap = createToolIdMappings(allTools);
  const convertedTools = await Promise.all(allTools.map(tool => {
    const toolId = onechatToLangchainIdMap.get(tool.id);
    return toolToLangchain({
      tool,
      logger,
      toolId,
      sendEvent,
      addReasoningParam
    });
  }));
  const reverseMappings = reverseMap(onechatToLangchainIdMap);
  return {
    tools: convertedTools,
    idMappings: reverseMappings
  };
};
exports.toolsToLangchain = toolsToLangchain;
const sanitizeToolId = toolId => {
  return toolId.replaceAll('.', '_').replace(/[^a-zA-Z0-9_-]/g, '');
};

/**
 * Create a [onechat tool id] -> [langchain tool id] mapping.
 *
 * Handles id sanitization (e.g. removing dot prefixes), and potential id conflict.
 */
exports.sanitizeToolId = sanitizeToolId;
const createToolIdMappings = tools => {
  const toolIds = new Set();
  const mapping = new Map();
  for (const tool of tools) {
    let toolId = sanitizeToolId(tool.id);
    let index = 1;
    while (toolIds.has(toolId)) {
      toolId = `${toolId}_${index++}`;
    }
    toolIds.add(toolId);
    mapping.set(tool.id, toolId);
  }
  return mapping;
};
exports.createToolIdMappings = createToolIdMappings;
const toolToLangchain = async ({
  tool,
  toolId,
  logger,
  sendEvent,
  addReasoningParam = true
}) => {
  const description = tool.getLlmDescription ? await tool.getLlmDescription({
    description: tool.description,
    config: tool.configuration
  }) : tool.description;
  const schema = await tool.getSchema();
  return (0, _tools.tool)(async (rawInput, config) => {
    let onEvent;
    if (sendEvent) {
      var _ref, _config$configurable$, _config$configurable, _config$toolCall;
      const toolCallId = (_ref = (_config$configurable$ = (_config$configurable = config.configurable) === null || _config$configurable === void 0 ? void 0 : _config$configurable.tool_call_id) !== null && _config$configurable$ !== void 0 ? _config$configurable$ : (_config$toolCall = config.toolCall) === null || _config$toolCall === void 0 ? void 0 : _config$toolCall.id) !== null && _ref !== void 0 ? _ref : 'unknown';
      const convertEvent = getToolEventConverter({
        toolCallId
      });
      onEvent = event => {
        sendEvent(convertEvent(event));
      };
    }

    // remove internal parameters before calling tool handler.
    const input = (0, _lodash.omit)(rawInput, ['_reasoning']);
    try {
      logger.debug(`Calling tool ${tool.id} with params: ${JSON.stringify(input, null, 2)}`);
      const toolReturn = await tool.execute({
        toolParams: input,
        onEvent
      });
      const content = JSON.stringify({
        results: toolReturn.results
      });
      logger.debug(`Tool ${tool.id} returned reply of length ${content.length}`);
      return [content, toolReturn];
    } catch (e) {
      logger.warn(`error calling tool ${tool.id}: ${e}`);
      logger.debug(e.stack);
      const errorToolReturn = {
        results: [(0, _onechatServer.createErrorResult)(e.message)]
      };
      return [`${e}`, errorToolReturn];
    }
  }, {
    name: toolId !== null && toolId !== void 0 ? toolId : tool.id,
    schema: addReasoningParam ? _zod.z.object({
      _reasoning: _zod.z.string().optional().describe('Brief reasoning of why you are calling this tool'),
      ...schema.shape
    }) : schema,
    description,
    verboseParsingErrors: true,
    responseFormat: 'content_and_artifact',
    metadata: {
      toolId: tool.id
    }
  });
};
exports.toolToLangchain = toolToLangchain;
const toolIdentifierFromToolCall = (toolCall, mapping) => {
  var _mapping$get;
  return (_mapping$get = mapping.get(toolCall.toolName)) !== null && _mapping$get !== void 0 ? _mapping$get : toolCall.toolName;
};
exports.toolIdentifierFromToolCall = toolIdentifierFromToolCall;
function reverseMap(map) {
  const reversed = new Map();
  for (const [key, value] of map.entries()) {
    if (reversed.has(value)) {
      throw new Error(`Duplicate value detected while reversing map: ${value}`);
    }
    reversed.set(value, key);
  }
  return reversed;
}
const getToolEventConverter = ({
  toolCallId
}) => {
  return toolEvent => {
    if (toolEvent.type === _onechatCommon.ChatEventType.toolProgress) {
      return {
        type: _onechatCommon.ChatEventType.toolProgress,
        data: {
          ...toolEvent.data,
          tool_call_id: toolCallId
        }
      };
    }
    throw new Error(`Invalid tool call type ${toolEvent.type}`);
  };
};