"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GET_RELEVANT_FIELD_NAMES_SYSTEM_MESSAGE = void 0;
exports.getRelevantFieldNames = getRelevantFieldNames;
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _inferenceCommon = require("@kbn/inference-common");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _common = require("../../../common");
var _concatenate_chat_completion_chunks = require("../../../common/utils/concatenate_chat_completion_chunks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const GET_RELEVANT_FIELD_NAMES_SYSTEM_MESSAGE = exports.GET_RELEVANT_FIELD_NAMES_SYSTEM_MESSAGE = `You are a helpful assistant for Elastic Observability. 
Your task is to determine which fields are relevant to the conversation by selecting only the field IDs from the provided list. 
The list in the user message consists of JSON objects that map a human-readable field "name" to its unique "id". 
You must not output any field names — only the corresponding "id" values. Ensure that your output follows the exact JSON format specified.`;
async function getRelevantFieldNames({
  index,
  start,
  end,
  dataViews,
  esClient,
  savedObjectsClient,
  chat,
  messages,
  signal,
  logger
}) {
  var _hasAnyHitsResponse$h, _hasAnyHitsResponse$h2;
  const dataViewsService = await dataViews.dataViewsServiceFactory(savedObjectsClient, esClient);
  const hasAnyHitsResponse = await esClient.search({
    index,
    _source: false,
    track_total_hits: 1,
    terminate_after: 1
  });
  const hitCount = typeof hasAnyHitsResponse.hits.total === 'number' ? hasAnyHitsResponse.hits.total : (_hasAnyHitsResponse$h = (_hasAnyHitsResponse$h2 = hasAnyHitsResponse.hits.total) === null || _hasAnyHitsResponse$h2 === void 0 ? void 0 : _hasAnyHitsResponse$h2.value) !== null && _hasAnyHitsResponse$h !== void 0 ? _hasAnyHitsResponse$h : 0;

  // all fields are empty in this case, so get them all
  const includeEmptyFields = hitCount === 0;
  const fields = await dataViewsService.getFieldsForWildcard({
    pattern: (0, _lodash.castArray)(index).join(','),
    allowNoIndex: true,
    includeEmptyFields,
    indexFilter: start && end ? {
      range: {
        '@timestamp': {
          gte: _datemath.default.parse(start).toISOString(),
          lt: _datemath.default.parse(end).toISOString()
        }
      }
    } : undefined
  });

  // else get all the fields for the found dataview
  const response = {
    indices: [index],
    fields: fields.flatMap(field => {
      var _field$esTypes;
      return ((_field$esTypes = field.esTypes) !== null && _field$esTypes !== void 0 ? _field$esTypes : [field.type]).map(type => {
        return {
          name: field.name,
          type
        };
      });
    })
  };
  const allFields = response.fields;
  const fieldNames = (0, _lodash.uniq)(allFields.map(field => field.name));
  const groupedFields = (0, _lodash.groupBy)(allFields, field => field.name);
  const shortIdTable = new _common.ShortIdTable();
  const MAX_CHUNKS = 5;
  const FIELD_NAMES_PER_CHUNK = 250;
  const fieldNamesToAnalyze = fieldNames.slice(0, MAX_CHUNKS * FIELD_NAMES_PER_CHUNK);
  const relevantFields = await Promise.all((0, _lodash.chunk)(fieldNamesToAnalyze, FIELD_NAMES_PER_CHUNK).map(async fieldsInChunk => {
    var _chunkResponse$messag, _chunkResponse$messag2, _chunkResponse$messag3, _chunkResponse$messag4;
    const chunkResponse$ = (await chat('get_relevant_dataset_names', {
      signal,
      stream: true,
      systemMessage: GET_RELEVANT_FIELD_NAMES_SYSTEM_MESSAGE,
      messages: [
      // remove the last function request
      ...messages.slice(0, -1), {
        '@timestamp': new Date().toISOString(),
        message: {
          role: _common.MessageRole.User,
          content: `Below is a list of fields. Each entry is a JSON object that contains a "name" (the field name) and an "id" (the unique identifier). Use only the "id" values from this list when selecting relevant fields:

            ${fieldsInChunk.map(fieldName => JSON.stringify({
            name: fieldName,
            id: shortIdTable.take(fieldName)
          })).join('\n')}`
        }
      }],
      functions: [{
        name: _common.SELECT_RELEVANT_FIELDS_NAME,
        description: `Return only the field IDs (from the provided list) that you consider relevant to the conversation. Do not use any of the field names. Your response must be in the exact JSON format:
              {
                "fieldIds": ["id1", "id2", "id3"]
              }
              Only include IDs from the list provided in the user message.`,
        parameters: {
          type: 'object',
          properties: {
            fieldIds: {
              type: 'array',
              items: {
                type: 'string'
              }
            }
          },
          required: ['fieldIds']
        }
      }],
      functionCall: _common.SELECT_RELEVANT_FIELDS_NAME
    })).pipe((0, _concatenate_chat_completion_chunks.concatenateChatCompletionChunks)(), (0, _rxjs.catchError)(error => {
      logger.error(`Encountered error running function ${_common.SELECT_RELEVANT_FIELDS_NAME}: ${JSON.stringify(error)}`);
      if ((0, _inferenceCommon.isInferenceRequestAbortedError)(error)) {
        // return empty fieldIds for chunk
        return (0, _rxjs.of)({
          message: {
            content: '',
            function_call: {
              name: _common.SELECT_RELEVANT_FIELDS_NAME,
              arguments: JSON.stringify({
                fieldIds: []
              })
            },
            role: 'assistant'
          }
        });
      }
      throw error;
    }));
    const chunkResponse = await (0, _rxjs.lastValueFrom)(chunkResponse$);
    return (_chunkResponse$messag = chunkResponse.message) !== null && _chunkResponse$messag !== void 0 && (_chunkResponse$messag2 = _chunkResponse$messag.function_call) !== null && _chunkResponse$messag2 !== void 0 && _chunkResponse$messag2.arguments ? JSON.parse(chunkResponse.message.function_call.arguments).fieldIds.map(fieldId => {
      const fieldName = shortIdTable.lookup(fieldId);
      return fieldName !== null && fieldName !== void 0 ? fieldName : fieldId;
    }).filter(fieldName => {
      return fieldsInChunk.includes(fieldName);
    }).map(field => {
      const fieldDescriptors = groupedFields[field];
      return `${field}:${fieldDescriptors.map(descriptor => descriptor.type).join(',')}`;
    }) : [(_chunkResponse$messag3 = (_chunkResponse$messag4 = chunkResponse.message) === null || _chunkResponse$messag4 === void 0 ? void 0 : _chunkResponse$messag4.content) !== null && _chunkResponse$messag3 !== void 0 ? _chunkResponse$messag3 : ''];
  }));
  return {
    fields: relevantFields.flat(),
    stats: {
      analyzed: fieldNamesToAnalyze.length,
      total: fieldNames.length
    }
  };
}