"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ChatState = void 0;
exports.createUseChat = createUseChat;
exports.useChat = useChat;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _react = require("react");
var _common = require("@kbn/kibana-utils-plugin/common");
var _common2 = require("../../common");
var _use_kibana = require("./use_kibana");
var _use_once = require("./use_once");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let ChatState = exports.ChatState = /*#__PURE__*/function (ChatState) {
  ChatState["Ready"] = "ready";
  ChatState["Loading"] = "loading";
  ChatState["Error"] = "error";
  ChatState["Aborted"] = "aborted";
  return ChatState;
}({});
function useChatWithoutContext({
  initialMessages,
  initialConversationId,
  notifications,
  service,
  chatService,
  connectorId,
  onConversationUpdate,
  onChatComplete,
  persist,
  disableFunctions,
  scopes
}) {
  const [chatState, setChatState] = (0, _react.useState)(ChatState.Ready);
  const systemMessage = (0, _react.useMemo)(() => {
    return chatService.getSystemMessage();
  }, [chatService]);
  (0, _use_once.useOnce)(initialMessages);
  (0, _use_once.useOnce)(initialConversationId);
  const [conversationId, setConversationId] = (0, _react.useState)(initialConversationId);
  const [messages, setMessages] = (0, _react.useState)(initialMessages);
  const [pendingMessages, setPendingMessages] = (0, _react.useState)();
  const abortControllerRef = (0, _react.useRef)(new AbortController());
  const onChatCompleteRef = (0, _react.useRef)(onChatComplete);
  onChatCompleteRef.current = onChatComplete;
  const onConversationUpdateRef = (0, _react.useRef)(onConversationUpdate);
  onConversationUpdateRef.current = onConversationUpdate;
  const handleSignalAbort = (0, _react.useCallback)(() => {
    setChatState(ChatState.Aborted);
  }, []);
  const handleError = (0, _react.useCallback)(error => {
    if (error instanceof _common.AbortError) {
      setChatState(ChatState.Aborted);
      return;
    }
    setChatState(ChatState.Error);
    if ((0, _common2.isTokenLimitReachedError)(error)) {
      setMessages(msgs => {
        var _error$meta, _error$meta2;
        return [...msgs, {
          '@timestamp': new Date().toISOString(),
          message: {
            content: _i18n.i18n.translate('xpack.observabilityAiAssistant.tokenLimitError', {
              defaultMessage: 'The conversation has exceeded the token limit. The maximum token limit is **{tokenLimit}**, but the current conversation has **{tokenCount}** tokens. Please start a new conversation to continue.',
              values: {
                tokenLimit: (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.tokenLimit,
                tokenCount: (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : _error$meta2.tokenCount
              }
            }),
            role: _common2.MessageRole.Assistant
          }
        }];
      });
      return;
    }
    notifications.toasts.addError(error, {
      title: _i18n.i18n.translate('xpack.observabilityAiAssistant.failedToLoadResponse', {
        defaultMessage: 'Failed to load response from the AI Assistant'
      })
    });
  }, [notifications.toasts]);
  const next = (0, _react.useCallback)(async (nextMessages, onError) => {
    // make sure we ignore any aborts for the previous signal
    abortControllerRef.current.signal.removeEventListener('abort', handleSignalAbort);

    // cancel running requests
    abortControllerRef.current.abort();
    abortControllerRef.current = new AbortController();
    setPendingMessages([]);
    setMessages(nextMessages);
    if (!connectorId || !nextMessages.length) {
      setChatState(ChatState.Ready);
      return;
    }
    setChatState(ChatState.Loading);
    const next$ = chatService.complete({
      getScreenContexts: () => service.getScreenContexts(),
      connectorId,
      messages: nextMessages,
      systemMessage,
      persist,
      disableFunctions: disableFunctions !== null && disableFunctions !== void 0 ? disableFunctions : false,
      signal: abortControllerRef.current.signal,
      conversationId,
      scopes
    });
    function getPendingMessages() {
      return [...completedMessages, ...(pendingMessage ? [(0, _lodash.merge)({
        message: {
          role: _common2.MessageRole.Assistant,
          function_call: {
            trigger: _common2.MessageRole.Assistant
          }
        }
      }, pendingMessage)] : [])];
    }
    const completedMessages = [];
    let pendingMessage;
    const subscription = next$.subscribe({
      next: event => {
        var _onConversationUpdate, _onConversationUpdate2;
        switch (event.type) {
          case _common2.StreamingChatResponseEventType.ChatCompletionChunk:
            if (!pendingMessage) {
              var _event$message$functi, _event$message$functi2;
              pendingMessage = {
                '@timestamp': new Date().toISOString(),
                message: {
                  content: event.message.content || '',
                  function_call: {
                    name: ((_event$message$functi = event.message.function_call) === null || _event$message$functi === void 0 ? void 0 : _event$message$functi.name) || '',
                    arguments: ((_event$message$functi2 = event.message.function_call) === null || _event$message$functi2 === void 0 ? void 0 : _event$message$functi2.arguments) || ''
                  }
                }
              };
            } else {
              var _event$message$functi3, _event$message$functi4;
              pendingMessage.message.content += event.message.content || '';
              pendingMessage.message.function_call.name += ((_event$message$functi3 = event.message.function_call) === null || _event$message$functi3 === void 0 ? void 0 : _event$message$functi3.name) || '';
              pendingMessage.message.function_call.arguments += ((_event$message$functi4 = event.message.function_call) === null || _event$message$functi4 === void 0 ? void 0 : _event$message$functi4.arguments) || '';
            }
            break;
          case _common2.StreamingChatResponseEventType.MessageAdd:
            pendingMessage = undefined;
            completedMessages.push(event.message);
            break;
          case _common2.StreamingChatResponseEventType.ConversationCreate:
            setConversationId(event.conversation.id);
            (_onConversationUpdate = onConversationUpdateRef.current) === null || _onConversationUpdate === void 0 ? void 0 : _onConversationUpdate.call(onConversationUpdateRef, event);
            break;
          case _common2.StreamingChatResponseEventType.ConversationUpdate:
            (_onConversationUpdate2 = onConversationUpdateRef.current) === null || _onConversationUpdate2 === void 0 ? void 0 : _onConversationUpdate2.call(onConversationUpdateRef, event);
            break;
        }
        setPendingMessages(getPendingMessages());
      },
      complete: () => {
        var _onChatCompleteRef$cu;
        setChatState(ChatState.Ready);
        const completed = nextMessages.concat(completedMessages);
        setMessages(completed);
        setPendingMessages([]);
        (_onChatCompleteRef$cu = onChatCompleteRef.current) === null || _onChatCompleteRef$cu === void 0 ? void 0 : _onChatCompleteRef$cu.call(onChatCompleteRef, completed);
      },
      error: error => {
        setPendingMessages([]);
        setMessages(nextMessages.concat(getPendingMessages()));
        onError === null || onError === void 0 ? void 0 : onError(error);
        handleError(error);
      }
    });
    abortControllerRef.current.signal.addEventListener('abort', () => {
      handleSignalAbort();
      subscription.unsubscribe();
    });
  }, [chatService, connectorId, conversationId, handleError, handleSignalAbort, persist, disableFunctions, service, systemMessage, scopes]);
  (0, _react.useEffect)(() => {
    return () => {
      abortControllerRef.current.abort();
    };
  }, []);
  const memoizedMessages = (0, _react.useMemo)(() => {
    return messages.concat(pendingMessages !== null && pendingMessages !== void 0 ? pendingMessages : []);
  }, [messages, pendingMessages]);
  const setMessagesWithAbort = (0, _react.useCallback)(nextMessages => {
    abortControllerRef.current.abort();
    setPendingMessages([]);
    setChatState(ChatState.Ready);
    setMessages(nextMessages);
  }, []);
  return {
    messages: memoizedMessages,
    setMessages: setMessagesWithAbort,
    state: chatState,
    next,
    stop: () => {
      abortControllerRef.current.abort();
    }
  };
}
function useChat(props) {
  const {
    services: {
      notifications
    }
  } = (0, _use_kibana.useKibana)();
  return useChatWithoutContext({
    ...props,
    notifications
  });
}
function createUseChat({
  notifications
}) {
  return parameters => {
    return useChatWithoutContext({
      ...parameters,
      notifications
    });
  };
}