"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.composeValidators = composeValidators;
exports.dictionaryValidator = dictionaryValidator;
exports.maxLengthValidator = maxLengthValidator;
exports.memoryInputValidator = memoryInputValidator;
exports.patternValidator = patternValidator;
exports.requiredValidator = requiredValidator;
exports.timeIntervalInputValidator = timeIntervalInputValidator;
var _mlParseInterval = require("@kbn/ml-parse-interval");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Provides a validator function for maximum allowed input length.
 * @param maxLength Maximum length allowed.
 */
function maxLengthValidator(maxLength) {
  return value => value && value.length > maxLength ? {
    maxLength: {
      requiredLength: maxLength,
      actualLength: value.length
    }
  } : null;
}

/**
 * Factory that provides a validator function for checking against pattern.
 * @param pattern Pattern to check against.
 * @returns A validator function that checks if the value matches the pattern.
 */
function patternValidator(pattern) {
  return value => pattern.test(value) ? null : {
    pattern: {
      matchPattern: pattern.toString()
    }
  };
}

/**
 * Factory that composes multiple validators into a single function.
 *
 * @param validators List of validators to compose.
 * @returns A validator function that runs all the validators.
 */
function composeValidators(...validators) {
  return value => {
    const validationResult = validators.reduce((acc, validator) => {
      return Object.assign(acc, validator(value) || {});
    }, {});
    return Object.keys(validationResult).length > 0 ? validationResult : null;
  };
}

/**
 * Factory to create a required validator function.
 * @returns A validator function that checks if the value is empty.
 */
function requiredValidator() {
  return value => {
    return value === '' || value === undefined || value === null ? {
      required: true
    } : null;
  };
}

/**
 * Type for the result of a validation.
 */

/**
 * Type for the result of a memory input validation.
 */

/**
 * Factory for creating a memory input validator function.
 *
 * @param allowedUnits Allowed units for the memory input.
 * @returns A validator function that checks if the value is a valid memory input.
 */
function memoryInputValidator(allowedUnits = _constants.ALLOWED_DATA_UNITS) {
  return value => {
    if (typeof value !== 'string' || value === '') {
      return null;
    }
    const regexp = new RegExp(`\\d+(${allowedUnits.join('|')})$`, 'i');
    return regexp.test(value.trim()) ? null : {
      invalidUnits: {
        allowedUnits: allowedUnits.join(', ')
      }
    };
  };
}

/**
 * Factory for creating a time interval input validator function.
 *
 * @returns A validator function that checks if the value is a valid time interval.
 */
function timeIntervalInputValidator() {
  return value => {
    if (value === '') {
      return null;
    }
    const r = (0, _mlParseInterval.parseInterval)(value);
    if (r === null) {
      return {
        invalidTimeInterval: true
      };
    }
    return null;
  };
}

/**
 * Factory to create a dictionary validator function.
 * @param dict Dictionary to check against.
 * @param shouldInclude Whether the value should be included in the dictionary.
 * @returns A validator function that checks if the value is in the dictionary.
 */
function dictionaryValidator(dict, shouldInclude = false) {
  const dictSet = new Set(dict);
  return value => {
    if (dictSet.has(value) !== shouldInclude) {
      return {
        matchDict: value
      };
    }
    return null;
  };
}