"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fieldsService = fieldsService;
var _app = require("../../common/constants/app");
var _error_wrapper = require("../client/error_wrapper");
var _fields_service_schema = require("./schemas/fields_service_schema");
var _fields_service = require("../models/fields_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getCardinalityOfFields(client, payload) {
  const fs = (0, _fields_service.fieldsServiceProvider)(client);
  const {
    index,
    fieldNames,
    query,
    timeFieldName,
    earliestMs,
    latestMs
  } = payload;
  return fs.getCardinalityOfFields(index, fieldNames, query, timeFieldName, earliestMs, latestMs);
}
function getTimeFieldRange(client, payload) {
  const fs = (0, _fields_service.fieldsServiceProvider)(client);
  const {
    index,
    timeFieldName,
    query,
    runtimeMappings,
    indicesOptions
  } = payload;
  return fs.getTimeFieldRange(index, timeFieldName, query, runtimeMappings, indicesOptions);
}

/**
 * Routes for fields service
 */
function fieldsService({
  router,
  routeGuard
}) {
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/fields_service/field_cardinality`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetFieldInfo']
      }
    },
    summary: 'Gets cardinality of fields',
    description: 'Returns the cardinality of one or more fields. Returns an Object whose keys are the names of the fields, with values equal to the cardinality of the field'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _fields_service_schema.getCardinalityOfFieldsSchema
      },
      response: {
        200: {
          body: _fields_service_schema.getCardinalityOfFieldsResponse,
          description: 'Cardinality of fields'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const resp = await getCardinalityOfFields(client, request.body);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/fields_service/time_field_range`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetFieldInfo']
      }
    },
    summary: 'Get time field range',
    description: 'Returns the time range for the given index and query using the specified time range.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _fields_service_schema.getTimeFieldRangeSchema
      },
      response: {
        200: {
          body: _fields_service_schema.getTimeFieldRangeResponse,
          description: 'Cardinality of fields'
        }
      }
    }
  }, routeGuard.basicLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const resp = await getTimeFieldRange(client, request.body);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}