"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JobImportService = void 0;
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _i18n = require("@kbn/i18n");
var _job_utils = require("../../../../../common/util/job_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function isImportedAdJobs(obj) {
  return Array.isArray(obj) && obj.some(o => o.job && o.datafeed);
}
function isDataFrameAnalyticsConfigs(obj) {
  return Array.isArray(obj) && obj.some(o => o.dest && o.analysis);
}
class JobImportService {
  constructor(esSearch, validateDatafeedPreview, getFilters) {
    this.esSearch = esSearch;
    this.validateDatafeedPreview = validateDatafeedPreview;
    this.getFilters = getFilters;
  }
  _readFile(file) {
    return new Promise((resolve, reject) => {
      if (file && file.size) {
        const reader = new FileReader();
        reader.readAsText(file);
        reader.onload = (() => {
          return () => {
            const data = reader.result;
            if (typeof data === 'string') {
              try {
                const json = JSON.parse(data);
                resolve(json);
              } catch (error) {
                reject();
              }
            } else {
              reject();
            }
          };
        })();
      } else {
        reject();
      }
    });
  }
  async readJobConfigs(file) {
    try {
      const json = await this._readFile(file);
      const jobs = Array.isArray(json) ? json : [json];
      if (isImportedAdJobs(jobs)) {
        const jobIds = jobs.map(j => j.job.job_id);
        return {
          jobs,
          jobIds,
          jobType: 'anomaly-detector'
        };
      } else if (isDataFrameAnalyticsConfigs(jobs)) {
        const jobIds = jobs.map(j => j.id);
        return {
          jobs,
          jobIds,
          jobType: 'data-frame-analytics'
        };
      } else {
        return {
          jobs: [],
          jobIds: [],
          jobType: null
        };
      }
    } catch (error) {
      return {
        jobs: [],
        jobIds: [],
        jobType: null
      };
    }
  }
  renameAdJobs(jobIds, jobs) {
    if (jobs.length !== jobs.length) {
      return jobs;
    }
    return jobs.map((j, i) => {
      const {
        jobId
      } = jobIds[i];
      j.job.job_id = jobId;
      j.datafeed.job_id = jobId;
      j.datafeed.datafeed_id = (0, _job_utils.createDatafeedId)(jobId);
      return j;
    });
  }
  async validateJobSourceIndices(indices) {
    if (!indices || indices.length === 0) {
      return [{
        index: undefined,
        error: _i18n.i18n.translate('xpack.ml.importExport.importFlyout.validation.sourceIndicesArrayEmpty', {
          defaultMessage: 'Source indices array is empty'
        })
      }];
    }
    const indexValidations = await Promise.all(indices.map(index => this.validateSingleIndex(index)));
    const invalidIndices = [];
    for (const result of indexValidations) {
      if (!result.valid) {
        invalidIndices.push({
          index: result.index,
          error: result.error
        });
      }
    }
    return invalidIndices;
  }
  async validateSingleIndex(index) {
    try {
      var _resp$_shards;
      const resp = await this.esSearch({
        index: [index],
        size: 0,
        body: {
          query: {
            match_all: {}
          }
        }
      });

      // Check if the index pattern matched any indices
      if (((_resp$_shards = resp._shards) === null || _resp$_shards === void 0 ? void 0 : _resp$_shards.total) === 0) {
        return {
          valid: false,
          index,
          error: _i18n.i18n.translate('xpack.ml.importExport.importFlyout.validation.indexPatternMatchesNoIndices', {
            defaultMessage: 'Index pattern matches no indices'
          })
        };
      }
      return {
        valid: true,
        index
      };
    } catch (error) {
      const errorMessage = (0, _mlErrorUtils.extractErrorMessage)(error);
      return {
        valid: false,
        index,
        error: errorMessage
      };
    }
  }
  renameDfaJobs(jobIds, jobs) {
    if (jobs.length !== jobs.length) {
      return jobs;
    }
    return jobs.map((j, i) => {
      const {
        jobId,
        destIndex
      } = jobIds[i];
      j.id = jobId;
      if (destIndex !== undefined) {
        j.dest.index = destIndex;
      }
      return j;
    });
  }
  async validateJobs(jobs, type) {
    const existingFilters = new Set((await this.getFilters()).map(f => f.filter_id));
    const tempJobs = [];
    const skippedJobs = [];
    const sourceIndicesErrors = new Map();
    const datafeedValidations = new Map();
    const commonJobs = type === 'anomaly-detector' ? jobs.map(j => ({
      jobId: j.job.job_id,
      indices: j.datafeed.indices,
      filters: getFilterIdsFromJob(j.job)
    })) : jobs.map(j => ({
      jobId: j.id,
      destIndex: j.dest.index,
      indices: Array.isArray(j.source.index) ? j.source.index : [j.source.index]
    }));
    if (type === 'data-frame-analytics') {
      const sourceIndexErrors = await Promise.all(commonJobs.map(({
        indices
      }) => this.validateJobSourceIndices(indices)));
      sourceIndexErrors.forEach((errors, i) => {
        if (errors.length > 0) {
          sourceIndicesErrors.set(commonJobs[i].jobId, errors);
        }
      });
    }
    if (type === 'anomaly-detector') {
      // exclude jobs with missing filters
      const validJobsForDatafeedCheck = jobs.filter(j => {
        var _commonJobs$find, _commonJobs$find$filt;
        return !((_commonJobs$find = commonJobs.find(cj => cj.jobId === j.job.job_id)) !== null && _commonJobs$find !== void 0 && (_commonJobs$find$filt = _commonJobs$find.filters) !== null && _commonJobs$find$filt !== void 0 && _commonJobs$find$filt.length);
      });
      const datafeedResults = await this.validateDatafeeds(validJobsForDatafeedCheck);
      datafeedResults.forEach(result => {
        datafeedValidations.set(result.jobId, result);
      });
    }
    commonJobs.forEach(({
      jobId,
      filters = [],
      destIndex
    }) => {
      const missingFilters = filters.filter(i => existingFilters.has(i) === false);
      const indicesErrors = sourceIndicesErrors.get(jobId);
      if (missingFilters.length === 0 && !indicesErrors) {
        tempJobs.push({
          jobId,
          ...(type === 'data-frame-analytics' ? {
            destIndex
          } : {})
        });
      } else {
        skippedJobs.push({
          jobId,
          ...(missingFilters.length > 0 ? {
            missingFilters
          } : {}),
          ...(indicesErrors ? {
            sourceIndicesErrors: indicesErrors
          } : {})
        });
      }
    });
    return {
      jobs: tempJobs,
      skippedJobs,
      datafeedValidations
    };
  }
  async validateDatafeeds(jobs) {
    const results = await Promise.all(jobs.map(async ({
      job,
      datafeed
    }) => {
      try {
        const combinedJob = {
          ...job,
          datafeed_config: datafeed
        };
        const response = await this.validateDatafeedPreview({
          job: combinedJob
        });
        if (response.error) {
          return {
            jobId: job.job_id,
            hasWarning: true,
            warningMessage: _i18n.i18n.translate('xpack.ml.jobsList.datafeedPreviewValidationFailed', {
              defaultMessage: `Unable to validate datafeed preview. Reason: {reason}`,
              values: {
                reason: (0, _mlErrorUtils.extractErrorMessage)(response.error)
              }
            })
          };
        }
        if (!response.valid) {
          return {
            jobId: job.job_id,
            hasWarning: true,
            warningMessage: _i18n.i18n.translate('xpack.ml.jobsList.datafeedPreviewFailed', {
              defaultMessage: 'Datafeed preview failed. This job may not run correctly.'
            })
          };
        }
        if (!response.documentsFound) {
          return {
            jobId: job.job_id,
            hasWarning: true,
            warningMessage: _i18n.i18n.translate('xpack.ml.jobsList.datafeedPreviewNoData', {
              defaultMessage: 'Datafeed preview returned no data. This job may not run correctly.'
            })
          };
        }
        return {
          jobId: job.job_id,
          hasWarning: false
        };
      } catch (error) {
        return {
          jobId: job.job_id,
          hasWarning: true,
          warningMessage: _i18n.i18n.translate('xpack.ml.jobsList.datafeedPreviewValidationFailed', {
            defaultMessage: `Unable to validate datafeed preview. Reason: {reason}`,
            values: {
              reason: (0, _mlErrorUtils.extractErrorMessage)(error)
            }
          })
        };
      }
    }));
    return results;
  }
}
exports.JobImportService = JobImportService;
function getFilterIdsFromJob(job) {
  const filters = new Set();
  job.analysis_config.detectors.forEach(d => {
    if (d.custom_rules === undefined) {
      return;
    }
    d.custom_rules.forEach(r => {
      if (r.scope === undefined) {
        return;
      }
      Object.values(r.scope).forEach(s => {
        filters.add(s.filter_id);
      });
    });
  });
  return [...filters];
}