"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.memoizedGetOrderedLinkList = exports.getOrderedLinkList = void 0;
exports.resolveLinkInfo = resolveLinkInfo;
exports.resolveLinks = resolveLinks;
exports.serializeResolvedLinks = serializeResolvedLinks;
var _lodash = require("lodash");
var _content_management = require("../../common/content_management");
var _external_link_tools = require("../components/external_link/external_link_tools");
var _dashboard_link_tools = require("../components/dashboard_link/dashboard_link_tools");
var _dashboard_link_strings = require("../components/dashboard_link/dashboard_link_strings");
var _links_strings = require("../components/links_strings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getOrderedLinkList = links => {
  return [...links].sort((linkA, linkB) => {
    return linkA.order - linkB.order;
  });
};

/**
 * Memoizing this prevents the links panel editor from having to unnecessarily calculate this
 * a second time once the embeddable exists - after all, the links component should have already
 * calculated this so, we can get away with using the cached version in the editor
 */
exports.getOrderedLinkList = getOrderedLinkList;
const memoizedGetOrderedLinkList = exports.memoizedGetOrderedLinkList = (0, _lodash.memoize)(links => {
  return getOrderedLinkList(links);
}, links => {
  return links;
});
function serializeResolvedLinks(resolvedLinks) {
  return resolvedLinks.map(({
    title,
    description,
    error,
    ...linkToSave
  }) => linkToSave).map(
  // fiilter out null values which may have been introduced by the session state backup (undefined values are serialized as null).
  link => Object.fromEntries(Object.entries(link).filter(([key, value]) => value !== null)));
}
async function resolveLinks(links = []) {
  const resolvedLinkInfos = await Promise.all(links.map(async link => {
    return {
      ...link,
      ...(await resolveLinkInfo(link))
    };
  }));
  return getOrderedLinkList(resolvedLinkInfos);
}
async function resolveLinkInfo(link) {
  if (link.type === _content_management.EXTERNAL_LINK_TYPE) {
    var _link$label;
    const info = {
      title: (_link$label = link.label) !== null && _link$label !== void 0 ? _link$label : link.destination
    };
    const {
      valid,
      message
    } = (0, _external_link_tools.validateUrl)(link.destination);
    if (valid) {
      return info;
    }
    return {
      ...info,
      error: new Error(message)
    };
  }
  if (link.type === _content_management.DASHBOARD_LINK_TYPE) {
    if (!link.destination) return {
      title: ''
    };
    try {
      const {
        attributes: {
          title,
          description
        }
      } = await (0, _dashboard_link_tools.fetchDashboard)(link.destination);
      return {
        label: link.label,
        title,
        description
      };
    } catch (error) {
      return {
        title: _dashboard_link_strings.DashboardLinkStrings.getDashboardErrorLabel(),
        description: error.message,
        error
      };
    }
  }
  throw new Error(_links_strings.LinksStrings.embeddable.getUnsupportedLinkTypeError());
}