"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LicensingPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/core/server");
var _register_analytics_context_provider = require("../common/register_analytics_context_provider");
var _license_update = require("../common/license_update");
var _routes = require("./routes");
var _services = require("./services");
var _licensing_route_handler_context = require("./licensing_route_handler_context");
var _on_pre_response_handler = require("./on_pre_response_handler");
var _plugin_status = require("./plugin_status");
var _license_fetcher = require("./license_fetcher");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @public
 * A plugin for fetching, refreshing, and receiving information about the license for the
 * current Kibana instance.
 */
class LicensingPlugin {
  constructor(context) {
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "isElasticsearchAvailable$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "loggingSubscription", void 0);
    (0, _defineProperty2.default)(this, "featureUsage", new _services.FeatureUsageService());
    (0, _defineProperty2.default)(this, "refresh", void 0);
    (0, _defineProperty2.default)(this, "license$", void 0);
    this.context = context;
    this.logger = this.context.logger.get();
    this.config = this.context.config.get();
  }
  setup(core) {
    this.logger.debug('Setting up Licensing plugin');
    const pollingFrequency = this.config.api_polling_frequency;
    const clientPromise = core.getStartServices().then(([{
      elasticsearch
    }]) => {
      return elasticsearch.client;
    });
    core.status.core$.pipe((0, _rxjs.map)(({
      elasticsearch
    }) => elasticsearch.level === _server.ServiceStatusLevels.available)).subscribe(this.isElasticsearchAvailable$);
    const {
      refresh,
      license$
    } = this.createLicensePoller(clientPromise, pollingFrequency.asMilliseconds());
    (0, _register_analytics_context_provider.registerAnalyticsContextProvider)(core.analytics, license$);
    core.status.set((0, _plugin_status.getPluginStatus$)(license$, this.stop$.asObservable()));
    core.http.registerRouteHandlerContext('licensing', (0, _licensing_route_handler_context.createRouteHandlerContext)(license$, core.getStartServices));
    const featureUsageSetup = this.featureUsage.setup();
    (0, _routes.registerRoutes)(core.http.createRouter(), featureUsageSetup, core.getStartServices);
    core.http.registerOnPreResponse((0, _on_pre_response_handler.createOnPreResponseHandler)(refresh, license$));
    this.refresh = refresh;
    this.license$ = license$;
    return {
      refresh,
      license$,
      featureUsage: featureUsageSetup
    };
  }
  createLicensePoller(clusterClient, pollingFrequency) {
    this.logger.debug(`Polling Elasticsearch License API with frequency ${pollingFrequency}ms.`);
    const isElasticsearchNotAvailable$ = this.isElasticsearchAvailable$.pipe((0, _rxjs.filter)(isElasticsearchAvailable => !isElasticsearchAvailable));

    // Trigger whenever the timer ticks or ES becomes available
    const intervalRefresh$ = this.isElasticsearchAvailable$.pipe((0, _rxjs.distinctUntilChanged)(), (0, _rxjs.filter)(isElasticsearchAvailable => isElasticsearchAvailable), (0, _rxjs.switchMap)(() => (0, _rxjs.timer)(0, pollingFrequency).pipe((0, _rxjs.takeUntil)(isElasticsearchNotAvailable$))), (0, _rxjs.throttleTime)(pollingFrequency) // avoid triggering too often
    );
    const licenseFetcher = (0, _license_fetcher.getLicenseFetcher)({
      clusterClient,
      logger: this.logger,
      cacheDurationMs: this.config.license_cache_duration.asMilliseconds(),
      maxRetryDelay: pollingFrequency
    });
    const {
      license$,
      refreshManually
    } = (0, _license_update.createLicenseUpdate)(intervalRefresh$, this.stop$, licenseFetcher);
    this.loggingSubscription = license$.subscribe(license => this.logger.debug(() => 'Imported license information from Elasticsearch:' + [`type: ${license.type}`, `status: ${license.status}`, `expiry date: ${(0, _moment.default)(license.expiryDateInMillis, 'x').format()}`].join(' | ')));
    return {
      refresh: async () => {
        this.logger.debug('Requesting Elasticsearch licensing API');
        return await refreshManually();
      },
      license$
    };
  }
  start() {
    if (!this.refresh || !this.license$) {
      throw new Error('Setup has not been completed');
    }
    return {
      refresh: this.refresh,
      getLicense: async () => await (0, _rxjs.firstValueFrom)(this.license$),
      license$: this.license$,
      featureUsage: this.featureUsage.start(),
      createLicensePoller: this.createLicensePoller.bind(this)
    };
  }
  stop() {
    this.stop$.next();
    this.stop$.complete();
    if (this.loggingSubscription !== undefined) {
      this.loggingSubscription.unsubscribe();
      this.loggingSubscription = undefined;
    }
  }
}
exports.LicensingPlugin = LicensingPlugin;