"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.uploadLicenseStatus = exports.uploadLicense = void 0;
var _reduxActions = require("redux-actions");
var _add_license = require("./add_license");
var _es = require("../../lib/es");
var _add_error_message = require("./add_error_message");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const uploadLicenseStatus = exports.uploadLicenseStatus = (0, _reduxActions.createAction)('LICENSE_MANAGEMENT_UPLOAD_LICENSE_STATUS');
const genericUploadError = _i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.genericUploadErrorMessage', {
  defaultMessage: 'Error encountered uploading license:'
});
const dispatchFromResponse = async (response, dispatch, currentLicenseType, newLicenseType, {
  history,
  licensing
}) => {
  const {
    error,
    acknowledged,
    license_status: licenseStatus,
    acknowledge
  } = response;
  if (error) {
    dispatch(uploadLicenseStatus({}));
    dispatch((0, _add_error_message.addUploadErrorMessage)(`${genericUploadError} ${error.reason}`));
  } else if (acknowledged) {
    if (licenseStatus === 'invalid') {
      dispatch(uploadLicenseStatus({}));
      dispatch((0, _add_error_message.addUploadErrorMessage)(_i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.invalidLicenseErrorMessage', {
        defaultMessage: 'The supplied license is not valid for this product.'
      })));
    } else if (licenseStatus === 'expired') {
      dispatch(uploadLicenseStatus({}));
      dispatch((0, _add_error_message.addUploadErrorMessage)(_i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.expiredLicenseErrorMessage', {
        defaultMessage: 'The supplied license has expired.'
      })));
    } else {
      const updatedLicense = await licensing.refresh();
      dispatch((0, _add_license.addLicense)(updatedLicense));
      dispatch(uploadLicenseStatus({}));
      history.replace('/home');
      // reload necessary to get left nav to refresh with proper links
      window.location.reload();
    }
  } else {
    // first message relates to command line interface, so remove it
    const messages = Object.values(acknowledge).slice(1);
    // messages can be in nested arrays
    const first = _i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.problemWithUploadedLicenseDescription', {
      defaultMessage: 'Some functionality will be lost if you replace your {currentLicenseType} license with a {newLicenseType} license. Review the list of features below.',
      values: {
        currentLicenseType: currentLicenseType.toUpperCase(),
        newLicenseType: newLicenseType.toUpperCase()
      }
    });
    dispatch(uploadLicenseStatus({
      acknowledge: true,
      messages: [first, ...messages]
    }));
  }
};
const uploadLicense = (licenseString, currentLicenseType, acknowledge) => async (dispatch, getState, services) => {
  dispatch(uploadLicenseStatus({
    applying: true
  }));
  let newLicenseType = null;
  try {
    ({
      type: newLicenseType
    } = JSON.parse(licenseString).license);
  } catch (err) {
    dispatch(uploadLicenseStatus({}));
    return dispatch((0, _add_error_message.addUploadErrorMessage)(_i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.checkLicenseFileErrorMessage', {
      defaultMessage: '{genericUploadError} Check your license file.',
      values: {
        genericUploadError
      }
    })));
  }
  try {
    const response = await (0, _es.putLicense)(services.http, licenseString, acknowledge);
    await dispatchFromResponse(response, dispatch, currentLicenseType, newLicenseType, services);
  } catch (err) {
    const message = err.responseJSON && err.responseJSON.error.reason ? err.responseJSON.error.reason : _i18n.i18n.translate('xpack.licenseMgmt.uploadLicense.unknownErrorErrorMessage', {
      defaultMessage: 'Unknown error.'
    });
    dispatch(uploadLicenseStatus({}));
    dispatch((0, _add_error_message.addUploadErrorMessage)(`${genericUploadError} ${message}`));
  }
};
exports.uploadLicense = uploadLicense;