"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isLangSmithEnabled = exports.getLangSmithTracer = void 0;
var _langsmith = require("langsmith");
var _tracer_langchain = require("@langchain/core/tracers/tracer_langchain");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns a custom LangChainTracer which adds the `exampleId` so Dataset 'Test' runs are written to LangSmith
 * If `exampleId` is present (and a corresponding example exists in LangSmith) trace is written to the Dataset's `Tests`
 * section, otherwise it is written to the `Project` provided
 *
 * @param apiKey API Key for LangSmith (will fetch from env vars if not provided)
 * @param projectName Name of project to trace results to
 * @param exampleId Dataset exampleId to associate trace with
 * @param logger
 */
const getLangSmithTracer = ({
  apiKey,
  projectName,
  exampleId,
  logger
}) => {
  try {
    if (!apiKey) {
      return [];
    }
    const lcTracer = new _tracer_langchain.LangChainTracer({
      projectName,
      // Shows as the 'test' run's 'name' in langsmith ui
      exampleId,
      client: new _langsmith.Client({
        apiKey
      })
    });
    return [lcTracer];
  } catch (e) {
    // Note: creating a tracer can fail if the LangSmith env vars are not set correctly
    logger.error(`Error creating LangSmith tracer: ${e.message}`);
  }
  return [];
};

/**
 * Returns true if LangSmith/tracing is enabled
 */
exports.getLangSmithTracer = getLangSmithTracer;
const isLangSmithEnabled = () => {
  try {
    // Just checking if apiKey is available, if better way to check for enabled that is not env var please update
    const config = _langsmith.Client.getDefaultClientConfig();
    return config.apiKey != null;
  } catch (e) {
    return false;
  }
};
exports.isLangSmithEnabled = isLangSmithEnabled;